#include "cit_io.h"
#include "krunpack.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>

#ifdef _WIN32
//#define CIT_UOFF_T      u64_t
//#define CIT_UOFF_T_CAST u64_t
//#define CIT_FMT_UOFF_T  "llu"
#define CIT_OFF_T       s64_t
//#define CIT_FMT_OFF_T   "lld"
#else // ndef _WIN32
//#define CIT_UOFF_T      off_t
//#define CIT_UOFF_T_CAST long long unsigned int
//#define CIT_FMT_UOFF_T  "llu"
#define CIT_OFF_T       off_t
//#define CIT_FMT_OFF_T   "lld"
#endif

#ifdef _WIN32
#define CIT_STRERROR() win_strerror()
#else
#define CIT_STRERROR() strerror(errno)
#endif

#define CAN_ADD_S32(a,b) ((((a)^(b))&0x80000000) || !((((s32_t) ((a)+(b))) ^ (a)) & 0x80000000))
#define CAN_ADD_S64(a,b) ((((a)^(b))&0x8000000000000000) || !((((s64_t) ((a)+(b))) ^ (a)) & 0x8000000000000000))

#define FMT_S64 "lld"

typedef struct cit_stat {
  u8_t is_dir;
  u8_t is_link;
  CIT_OFF_T fsize;
} cit_stat_t;


static citerr_t cit_stat_file (char *filename, cit_stat_t *stat_out);
static citerr_t fread_fully (u8_t *buf,
                             size_t size,
                             FILE *f);
static citerr_t cit_mkdir_2 (char *path);
static citerr_t cit_mkdir(char *path);
static citerr_t cit_getpath (char *path, char **output);
static citerr_t mkdir_for_file (char *filename);
static citerr_t fwrite_fully(u8_t *buf,
                             size_t size,
                             char *filename,
                             FILE *f);
static void sleep_ms (u32_t ms);
//static FILE *cit_fopen (char *filename, char *mode);
#ifdef _WIN32
char *win_strerror(void);
#define WIN_STRERROR_BUF_LEN 1024
static char win_strerror_buf[WIN_STRERROR_BUF_LEN];
#endif

#ifdef _WIN32
char *win_strerror(void) {
  memset(win_strerror_buf, 0, WIN_STRERROR_BUF_LEN + 1);
  strerror_s(win_strerror_buf, WIN_STRERROR_BUF_LEN, errno);
  return win_strerror_buf;
}
#endif

citerr_t cit_read_file (char *filename,
                        u8_t **payload_out,
                        size_t *len_out) {
                           
  // *payload_out must be freed by the caller if the result is CE_OK
  
  cit_stat_t sb;
  
  citerr_t ecode = CE_OK;
  FILE *fp = NULL;
  s64_t mask = 0;
  
  *payload_out = NULL;
  *len_out = 0;

  // stat the file once before opening it,
  // to check it isn't a directory or something
  memset(&sb, 0, sizeof(cit_stat_t));
  if (CE_OK != (ecode = cit_stat_file(filename, &sb))) {
    printf("File to be loaded is missing: \"%s\".\n", filename);
    return ecode;
  }

  if (sb.is_dir) {
    printf("Could not load file \"%s\", because it is "
              "actually a directory.\n", filename);
    return CE_FILE_NOT_FILE;
  }
  
  // then, open it
  if (NULL == (fp = cit_fopen (filename, "rb")))
    return CE_FOPEN_READ_FILE;

  // and now we stat it again
  memset(&sb, 0, sizeof(cit_stat_t));
  if (CE_OK != (ecode = cit_stat_file(filename, &sb))) {
    fclose(fp);
    return ecode;
  }

  if (sb.is_dir) {
    printf("Could not load file \"%s\", because it is "
                 "actually a directory.\n", filename);
    fclose(fp);
    return CE_FILE_NOT_FILE;
  }
  
  if (sb.fsize==0) {
    fclose(fp);
    return CE_OK;
  }

  if (sizeof(size_t) == 4)
    mask = 0x7fffffff;
  else
    mask = 0x7fffffffffffffff;

//printf("cit_read_file: %s, allocing %" FMT_SIZET " bytes\n", filename, sb.fsize);

  if (   ((sizeof(sb.fsize) == 4) && ( ! CAN_ADD_S32 (sb.fsize, 1) ))
      || ((sizeof(sb.fsize) == 8) && ( ! CAN_ADD_S64 (sb.fsize, 1) ))) {
    if (sizeof(sb.fsize) == 4) {
      printf("File \"%s\" was too large to open: maximum size %" FMT_S64 ", file size %" FMT_S64 "\n",
                filename, mask-1, (s64_t) sb.fsize);
      fclose(fp);
      return CE_FILE_TOOLARGE;
    } else {
      printf("File \"%s\" was too large to open: maximum size %" FMT_S64 ", file size %" FMT_S64 "\n",
                filename, mask-1, (s64_t) sb.fsize);
      fclose(fp);
      return CE_FILE_TOOLARGE;
    }
  }

  *payload_out = malloc ((sb.fsize + 1) & ((size_t) mask));
  (*payload_out)[sb.fsize] = 0; // allocate an extra byte at the end of the block and set it to 0

  if (sizeof(off_t) == 4 || sizeof(size_t) == 4)
    mask = 0x7fffffff;
  else
    mask = 0x7fffffffffffffff;

  ecode = fread_fully (*payload_out, sb.fsize & ((size_t) mask), fp);
  fclose(fp);
  
  if (CE_OK != ecode) {
    free(*payload_out);
    *payload_out = NULL;
    return ecode;
  }
  
  *len_out = sb.fsize & ((size_t) mask);

  return ecode;
  
}


// FIXME: pass in filename for errmsg
static citerr_t fread_fully (u8_t *buf,
                             size_t size,
                             FILE *f) {
  size_t size_read=0;
  char *errmsg = NULL;
  while (size_read<size) {
    size_t j;
    size_read += (j = fread (buf+size_read, 1, size-size_read, f));
    if (ferror(f)) {
      errmsg = CIT_STRERROR();
      printf("Problem reading file: %s.\n", errmsg);
      return CE_FREAD;
    } else if (feof(f)) {
      return CE_FREAD_EOF;
    }
    if (!j) {
      sleep_ms(1);
    }
  }
  return CE_OK;
}



// FIXME: add follow_link parameter, choose between stat() and lstat()
citerr_t cit_stat_file (char *filename, cit_stat_t *stat_out) {
#ifdef _WIN32
  WIN32_FILE_ATTRIBUTE_DATA fad;
  u64_t tmpu64;
#else
  struct stat sb;
#endif

#ifdef _WIN32
  memset(&fad, 0, sizeof(WIN32_FILE_ATTRIBUTE_DATA));
#else
  memset(&sb, 0, sizeof(struct stat));
#endif

  memset(stat_out, 0, sizeof(cit_stat_t));

#ifdef _WIN32
  if (!GetFileAttributesEx(filename, GetFileExInfoStandard, &fad))
    return CE_STAT;
//printf("file attributes(%s) = 0x%x\n", filename, fad.dwFileAttributes);
  if (fad.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
//printf("is_dir=1\n");
    stat_out->is_dir = 1;
  }
  tmpu64 = fad.nFileSizeHigh;
  tmpu64 <<= 32;
  tmpu64 &= 0xffffffff00000000UL;
  tmpu64 |= 0xffffffff & fad.nFileSizeLow;
  stat_out->fsize = (off_t) 0x7fffffffffffffff & tmpu64;
#else // ndef _WIN32
  if (lstat(filename, &sb))
    return CE_STAT;
  if ((sb.st_mode & S_IFMT) == S_IFLNK)
    stat_out->is_link = 1;
  if (stat(filename, &sb))
    return CE_STAT;
  if ((sb.st_mode & S_IFMT) == S_IFDIR)
    stat_out->is_dir = 1;

  stat_out->fsize = sb.st_size;
#endif
  return CE_OK;
}

static void sleep_ms (u32_t ms) {
#ifdef _WIN32
  Sleep(ms);
#else
  usleep(ms*1000);
#endif
}


FILE *cit_fopen (char *filename, char *mode) {
  
  FILE *f = NULL;
  u8_t failed=0;
  char *errmsg = NULL;

#ifdef _WIN32
  if (fopen_s (&f, filename, mode)) {
    failed = 1;
  }
#else
  if (NULL == (f = fopen (filename, mode))) {
    failed = 1;
  }
#endif
  if (failed) {
    errmsg = CIT_STRERROR();
    printf("ERROR: Could not open file \"%s\" (%s).\n", filename, errmsg);
  }
  return f;
}



citerr_t cit_write_file (char *filename, u8_t overwrite,
                         u8_t *payload, size_t len, u8_t mkpath) {

  cit_stat_t sb;

  FILE *fp = NULL;
  citerr_t ecode = CE_OK;
  char *symlink_text = " ";

  // general file writing scheme
  // does the file exist? if so, do we have permission to overwrite it?
  memset(&sb, 0, sizeof(cit_stat_t));
  if (CE_OK == cit_stat_file(filename, &sb)) {
    // stat succeeded: file exists
    if (overwrite) {
      // but we are cleared to overwrite it
      if (sb.is_dir || sb.is_link) {
        // but we can't, because it's a directory or symlink
#ifndef _WIN32
        symlink_text = " a symbolic link or ";
#endif
        printf("ERROR: Could not overwrite existing file \"%s\" because it is "
               "not actually a file. It is"
               "%s"
               "a directory.\n",
               filename,
               symlink_text);
        return CE_FILE_NOT_FILE;
      }
    } else {
      printf("ERROR: Refusing to overwrite file \"%s\".\n", filename);
      return CE_FILE_EXISTS;
    }
  }

  if (mkpath) {
    if (CE_OK != (ecode = mkdir_for_file (filename))) {
      return ecode;
    }
  }
  
  // write the payload
  if (NULL == (fp = cit_fopen (filename, "wb"))) {
    /*
    errmsg = CIT_STRERROR(cit);
    CIT_EPRINTF(cit->tw, 7,
                 "ERROR: Could not write file \"%s\" (%s).\n",
                 filename, errmsg);
    */
    return CE_FOPEN_OPFILE;
  }
  ecode = fwrite_fully (payload, len, filename, fp);
  fclose(fp);
  return ecode;
}



static citerr_t fwrite_fully(u8_t *buf,
                             size_t size,
                             char *filename,
                             FILE *f) {
  size_t size_written=0;
  char *errmsg = NULL;
  while (size_written < size) {
    size_t j;
//printf("buf=%p\n", buf);
//printf("fwrite(%p, 1, %zu, [%s])\n", buf+size_written, size-size_written, filename);
    size_written+=(j=fwrite(buf+size_written,1,size-size_written,f));
    
    if (ferror(f)) {
      errmsg = CIT_STRERROR();
      printf("ERROR: Something went wrong writing file %s (%s).\n", filename, errmsg);
      return CE_FWRITE;
    } else if (feof(f)) {
      errmsg = CIT_STRERROR();
      printf("ERROR: An unexpected EOF was encountered while writing file %s (%s).\n", filename, errmsg);
      return CE_FREAD_EOF;
    }
    if (!j) {
      sleep_ms(1);
    }
  }
  return CE_OK;
}



// pass in a filename and it will make a directory
// to contain that file

citerr_t mkdir_for_file (char *filename) {
  char *dirname = NULL;
  citerr_t ecode = CE_OK;
  // strip filename off path
  ecode = cit_getpath (filename, &dirname);
//printf("[%u] %s -> %s\n", ecode, filename, dirname);
  switch (ecode) {
    case CE_OK:
      // have path, make directory
      ecode = cit_mkdir (dirname);
      free(dirname);
      break;
    case CE_GETPATH_NOSEP:
      // no path, just filename, no mkdir needed
      ecode = CE_OK;
      break;
    default:
      // error
      return ecode;
  }
  return ecode;
}



citerr_t cit_getpath (char *path, char **output) {
  // *output must be freed by the caller if there is no error
  size_t j;
  size_t pathlen=0;
  u8_t nuke=0;
  *output = NULL;
  pathlen = strlen(path);
  if (!pathlen) {
    printf("BUG: path=\"\"\n");
    return CE_GETPATH_ZL;
  }
  *output = malloc(1 + pathlen);
  memcpy(*output, path, 1 + pathlen);
  // find last slash
  for (j=pathlen;j>0;j--) { // goes from L to 1, not (L-1) to 0
    nuke = ((*output)[j-1] == '/'); // is it a pathsep?
#ifdef _WIN32
    if ((*output)[j-1] == '\\')
      nuke = 1;
#endif
    //if ((*output)[j-1] == CIT_PATHSEP) {
    if (nuke) {
      // got it, nuke it!
      (*output)[j-1] = 0; // null terminator
      break;
    }
  }
  if (!j) {
    // no slashes found -- fail
    free(*output);
    *output = NULL;
    return CE_GETPATH_NOSEP;
  }
  return CE_OK;
}



static citerr_t cit_mkdir(char *path) {

  // FIXME: probably not secure

  // recursively build a directory path
  // start at the beginning of the path and for each fragment:
  // - does the fragment exist on disc? if it does, make sure
  //   it is a directory and not a file
  // - if it doesn't, create the directory and carry on
  size_t i;
  size_t len;
  cit_stat_t sb;
  size_t last_pathsep_index=0;
  
  char *path_clone = NULL;
  citerr_t ecode = CE_OK;
  
  len = strlen(path);
  path_clone = malloc(len+1);
  memcpy(path_clone, path, len+1);
  
  // get rid of any multiple slashes (UNIX only, don't bother on windows)
#ifndef _WIN32
  for (i=0;i<len+1;i++) {
    size_t j,k;
    j=i;
    while (j<len && (path_clone[j] == CIT_PATHSEP))
      j++;
    if (j >= len)
      j = len - 1;
    k=i;
    if (k >= len - 1)
      k = len - 2;
    if (j!=k) {
      memmove(path_clone+k+1, path_clone+j, len+1-j);
      path_clone[k+1+len-j] = 0; // null-terminate
    }
  }
  
#ifdef CIT_DBG
  printf("cit_mkdir path after demultislash: %s\n", path_clone);
#endif
#endif
  
  for (i=0;i<len+1;i++) {

    //size_t j=0;
    u8_t is_pathsep=0;
    char *not_a_directory_fmtstg = "";
    
    //if (path_clone[i] == CIT_PATHSEP || !path_clone[i]) {
    is_pathsep = (path_clone[i] == '/');
#ifdef _WIN32
    if (path_clone[i] == '\\')
      is_pathsep = 1;
#endif
    if (is_pathsep || !path_clone[i]) {

#ifdef _WIN32
      // is it the drive letter?
      if (i == 2 && isalpha(path_clone[0]) && path_clone[1] == ':') {
        last_pathsep_index = i;
        continue;
      }
#endif
      
      if (!i) // leading slash, don't try to create the root directory
        continue;
      
      path_clone[i] = 0; // edit string: insert null terminator

      // OK so this is complicated slightly by the fact that
      // we can have paths like ./../../././beans/../oats/./ etc
      // but basically we never try to create a '.' or a '..' so
      // we need to detect those.

      if ((last_pathsep_index+1 < len)
          && ((!strcmp(path_clone+last_pathsep_index+1, "."))
              || !strcmp(path_clone+last_pathsep_index+1, ".."))) {
        // don't try to create these
        last_pathsep_index = i;
#ifdef CIT_DBG
        printf("cit_mkdir: will not create %s\n", path_clone);
#endif
        path_clone[i] = CIT_PATHSEP; // remove null terminator again
        continue;
      }
        
      // check whether it exists
      //memset(&sb, 0, sizeof(cit_stat_t));
      if (CE_OK == cit_stat_file(path_clone, &sb)) {

//printf("cit_stat_file(): path_clone=%s\n", path_clone);

        // it exists
        //if (!(statbuf.st_mode & S_IFDIR) && !(statbuf.st_mode & symlink_bit)) {
        if ((!sb.is_dir) && (!sb.is_link)) {
          // uh oh, it's not a directory or symlink
#ifdef _WIN32
          not_a_directory_fmtstg = "ERROR: Trying to create path, but \"%s\" is not a directory "
                                   "or symbolic link.\n";
#else
          not_a_directory_fmtstg = "ERROR: Trying to create path, but \"%s\" is not a directory.\n";
#endif
          printf(not_a_directory_fmtstg, path_clone);
          ecode = CE_FILE_IN_PATH;
          goto L_cit_mkdir_end;
        }
      } else {
        if (CE_OK != (ecode = cit_mkdir_2 (path_clone)))
          goto L_cit_mkdir_end;
//#ifdef CIT_DBG
        printf("  + Created directory: %s\n", path_clone);
//#endif
      }
      
      path_clone[i] = CIT_PATHSEP; // remove null terminator again
      last_pathsep_index = i;
    }
  }
L_cit_mkdir_end:
  if (path_clone)
    free(path_clone);
  return ecode;
}



static citerr_t cit_mkdir_2 (char *path) {
  char *errmsg = NULL;
#ifdef CIT_DBG
  printf("cit_mkdir_2(%s)\n", path);
#endif
#ifdef _WIN32
  if (_mkdir (path)) {
#else
  if (mkdir (path, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH)) { // 755
#endif
    errmsg = CIT_STRERROR();
    printf("ERROR: Could not create directory \"%s\" (%s). This may be because one "
             "of the path fragments is actually a file, or it may be a permissions "
             "problem.\n",
             path, errmsg);
    return CE_MKDIR;
  }
  return CE_OK;
}

