
//#include "crunpack.h"

#include "roomdata.h"
#include "citconf.h"
#include "world.h"
#include "tiledata.h"
#include "vram.h"
#include "parse.h"
#include "tilegfx.h"
#include "triangle.h"
#include "raster.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
  
#define ROOM_NAME_MAX_LEN 255

static void trace (char *s);
static void trace2(char *s);
// static void trace3(char *s);

static citerr_t room_name_append (char *buf, size_t *fill, u8_t c);

static citerr_t room_name_append_s (char *buf, size_t *fill, char *s, size_t s_len);

static citerr_t parse_room_name (char room_name[ROOM_NAME_MAX_LEN+1],
                                 room_data_t *roomd,
                                 world_data_t *world);
                                 
static citerr_t room_flood_tile  (u8_t tile_id,
                                  tile_data_t *tiledata,
                                  vram_t *vram,
                                  u8_t pixel_base_pairs[WORLD_NUM_PXPAIRS],
                                  u8_t animation_speeds[WORLD_NUM_ANIMSPEEDS]);
                                  
static void print_room_features (u8_t b);

static citerr_t room_n3x  (room_data_t *room,
                           tile_data_t *tiledata,
                           u8_t *room_features_inout,
                           u8_t star_port_destroyed,
                           u8_t well_drained,
                           u8_t ice_crystal,
                           u8_t crystals[NUM_CRYSTALS],
                           u8_t lab_opened,
                           u8_t battlements_lift_on,
                           u8_t tiles_to_blit,
                           u8_t room_id,
                           u8_t pixel_base_pairs[WORLD_NUM_PXPAIRS],
                           u8_t animation_speeds[WORLD_NUM_ANIMSPEEDS],
                           vram_t *vram,
                           s16_t *sm_tactile_x_out,
                           s16_t *sm_tactile_y_out);
                           
static void room_byte_to_coordinates (u8_t rb, u8_t *x_out, u8_t *y_out);

static citerr_t room_rectangles (room_data_t *room, // Lrectangles
                                 s8_t num_rectangles,
                                 u8_t room_id,
                                 u8_t *water_X1_out,
                                 u8_t *water_X2_out,
                                 u8_t *water_level_out,
                                 u8_t star_port_destroyed,
                                 u8_t well_drained,
                                 vram_t *vram);
                                 
static citerr_t room_triangles (room_data_t *room,
                                u8_t star_port_destroyed,
                                u8_t *num_triangles_inout, // pass in from (decor_bar & 0x7)
                                triangle_t trgls_inout[MAX_TRIANGLES]);

static citerr_t fake_draw_triangles (triangle_t *tgls, s8_t nt, vram_t *vram, u8_t outlines_only);

static citerr_t room_item_pad (room_data_t *room,
                               s16_t item_tile_id,
                               u8_t *item_pad_X_out,
                               u8_t *item_pad_Y_out,
                               u8_t plot_mode,
                               tile_data_t *tiledata,
                               vram_t *vram,
                               u8_t *pixel_base_pairs,
                               u8_t *animation_speeds);
                              
static citerr_t tile_plot_fix_coords_tile_slot_0 (tile_data_t *tiledata,
                                                  tile_slot_t *ts0,
                                                  u8_t t_500[T_500_LEN],
                                                  u8_t plot_mode,
                                                  u8_t x,
                                                  u8_t y,
                                                  vram_t *vram);
                                                  
static citerr_t get_num_ropes_and_ladders(room_data_t *room,
                                          u8_t *num_ropes_out,
                                          u8_t *num_ladders_out);
                                          
static citerr_t room_ropes  (room_data_t *room,
                             u8_t num_ropes,
                             vram_t *vram);
                             
static citerr_t draw_ladder_rung (s8_t y, u8_t c, u16_t vp, vram_t *vram);

static citerr_t room_ladders(room_data_t *room, //char *s,
                      u8_t num_ladders,
                      u8_t *pixel_base_pairs,
                      vram_t *vram);
                      
static u8_t do_shifter (u8_t t_shifter_inout[6], u8_t *carry_out);

static citerr_t room_stars (vram_t *vram);

static citerr_t room_flask (u8_t flask_id,
                            tile_data_t *tiledata,
                            vram_t *vram,
                            u8_t pixel_base_pairs[WORLD_NUM_PXPAIRS],
                            u8_t animation_speeds[WORLD_NUM_ANIMSPEEDS],
                            u8_t flasks_positions[WORLD_NUM_FLASKS],
                            u8_t *flask_x_out,
                            u8_t *flask_y_out);
                            
static citerr_t room_door  (room_data_t *room,
                            tile_data_t *tiledata,
                            vram_t *vram,
                            u8_t *pixel_base_pairs,
                            u8_t *animation_speeds,
                            u8_t unlocked_mode,
                            s16_t *door_x_coord_out,
                            s16_t *door_y_coord_out);
                            
static citerr_t room_pillars  (room_data_t *room,
                               tile_data_t *tiledata,
                               u8_t *pixel_base_pairs,
                               u8_t *animation_speeds,
                               vram_t *vram);
                               
static void map_collision_colour (u8_t col_colour, u8_t *r_out, u8_t *g_out, u8_t *b_out);
                               
                               /*
static citerr_t draw_line(s16_t v1x,
                          s16_t v1y,
                          s16_t v2x,
                          s16_t v2y,
                          u8_t r,
                          u8_t g,
                          u8_t b,
                          u8_t c,
                          vram_t *vram); */

// ---

static void trace (char *s) { }
static void trace2(char *s) { }
// static void trace3(char *s) { }

static citerr_t room_name_append (char *buf, size_t *fill, u8_t c) {
  if (*fill >= ROOM_NAME_MAX_LEN) {
    printf("\nRoom name buffer overflow\n");
    return CE_ROOM_NAME_BUF_OVERFLOW;
  }
  buf[*fill] = (char) c;
  (*fill)++;
  return CE_OK;
}

static citerr_t room_name_append_s (char *buf, size_t *fill, char *s, size_t s_len) {
  size_t i;
  citerr_t e;
  for (i=0; i < s_len; i++) {
    e = room_name_append(buf, fill, s[i]);
    if (CE_OK != e) { return e; }
  }
  return CE_OK;
}

static citerr_t parse_room_name (char room_name[ROOM_NAME_MAX_LEN+1],
                                 room_data_t *roomd,
                                 world_data_t *world) {

  s16_t i;
  citerr_t e;
  u8_t c;
  size_t fill;
  s8_t name_ix;
  size_t z;

  memset(room_name, 0, ROOM_NAME_MAX_LEN+1);
  
  // ROOM NAME
  
  i = 1;
  fill=0;
  
  e = room_data_read (roomd, &c);
  if (CE_OK != e) { return CE_TRUNC_ROOM_NAME; }
  
  if ( ! (0x80 & c) ) {
  
    //room_name_msg = "  + Room name (inline): ";
    printf("  + Room name (inline): ");
  
    do { // 347a: P_push_inline_name
    
      e = room_name_append(room_name, &fill, c);
      if (CE_OK != e) { return e; }
      
      if ( ! (c & 0x80) ) {
        e = room_data_read (roomd, &c);
        if (CE_OK != e) { return CE_TRUNC_INLINE_ROOM_NAME; }
      }
      
    } while ( (c != 0) && ! (c & 0x80) );
    
    // don't understand why this is needed, but
    if (c != 0) {
      e = room_name_append (room_name, &fill, c & 0x7f);
      if (CE_OK != e) { return e; }
    }

  } else {
  
  /*
for (i=0; i < WORLD_GLOBAL_ROOM_TBL_LEN; i++) {
  printf ("%02x ", world->room_name_page[i]);
  if (! (i & 0xf)) { printf ("\n"); }
} */
  
    // global room name
    i = -1;
    name_ix = c & 0x7f;
    
    //room_name_msg = "  + Room name (global, entry #".name_ix."): ";
    printf("  + Room name (global, entry #%u): ", name_ix);
    
    do { // 348b: P_find_room_name
    
      i++;
      //if (i == 256) { i=0; } // hack for s16_t rather than u8_t
      
      if (i >= WORLD_GLOBAL_ROOM_TBL_LEN) {
        printf("\nERROR: ran off end of room_name_page [1]\n");
        return CE_ROOM_NAME_PAGE_OVERFLOW;
      }
      c = world->room_name_page[i];
      if (c & 0x80) {
        name_ix--;
      }
//printf("i = %d, name_ix = %u\n", i, name_ix);
    } while (name_ix >= 0);
    
    // c now contains a terminated byte
    
    do { // 3494: P_push_room_name
      e = room_name_append(room_name, &fill, c);
      if (CE_OK != e) { return e; }
      i++;
      if (i >= WORLD_GLOBAL_ROOM_TBL_LEN) {
        printf("\nERROR: ran off end of room_name_page [2]\n");
        return CE_ROOM_NAME_PAGE_OVERFLOW;
      }
      c = world->room_name_page[i];
    } while ( (c != 0) && ! (c & 0x80) );
    
  }
  
  if (c == 0) {
    e = room_name_append_s (room_name, &fill, " ehT", 4);
    if (CE_OK != e) { return e; }
  }
  
  // eliminate terminator bit
  room_name[0] &= 0x7f;
  
  // reverse it
  for (z=0; z < (fill / 2); z++) {
    char a;
    a = room_name[z];
    room_name[z] = room_name[fill-(1+z)];
    room_name[fill-(1+z)] = a;
  }
  
  //room_name_msg .= "\"".room_name."\"";
  printf("\"%s\"\n", room_name);
  
  return CE_OK;
  
}

  // sequence is as follows:
  
  // 1.  tile flood fill
  // 2.  N3X tile decor, round one (max. 7 tiles)
  // 3.  rectangles (max. 31)
  // 4.  triangles (max. 7)
  // 5.  item pad + item on pad
  // 6.  N3X tile decor, round two (max. 31 tiles)
  // 7.  ropes
  // 8.  ladders
  // 9.  stars
  // 10. flasks
  // 11. doors
  // 12. pillars
  // 13. (animated sprites TBC)
  
citerr_t parse_room (s16_t room_id,
                     room_data_t *roomd,
                     tile_data_t *tiledata,
                     citadel_config_t *config,
                     world_data_t *world,
                     s16_t *flask_x_out,
                     s16_t *flask_y_out,
                     vram_t *vram,
                     s16_t *sm_tactile_x_out,
                     s16_t *sm_tactile_y_out) {
                     
  // u16_t pos;
  // u8_t item_2_current_room;
  citerr_t e;
  char room_name[ROOM_NAME_MAX_LEN+1];
  u8_t flood_tile_id;
  u8_t room_features;
  u8_t num_tiles_round1_round2;
  u8_t round_1_num_tiles, round_2_num_tiles;
  u8_t b;
  s8_t num_rectangles;
  u8_t num_triangles;
  u8_t skip_trgl_stuff;
  u8_t water_X1, water_X2, water_level;
  triangle_t triangles[MAX_TRIANGLES];
  u8_t i;
  u8_t item_pad_X, item_pad_Y;
  u8_t have_ropes_ladders;
  u8_t num_ropes, num_ladders;
  s16_t flask_id;
  u8_t flask_x8, flask_y8;
  s16_t door_x_coord, door_y_coord;

  *flask_x_out = -1;
  *flask_y_out = -1;
  flask_x8 = 0xff;
  flask_y8 = 0xff;

  // pos = 0;
  
//printf("first byte is %x, next %x\n", roomd->buf[roomd->pos], roomd->buf[1+roomd->pos]);
  
  printf("\nRoom %d parse:\n", room_id);
  
  room_data_reset (roomd);
  
  // FIXME: what value should this be initialised to?
  // item_2_current_room = 0;
  
  e = parse_room_name(room_name, roomd, world);
  if (CE_OK != e) { return e; }

  trace ("352c");
  
  //printf("%s\n", room_name_msg);
  
  // handles flood fill tile:
  e = room_data_read(roomd, &flood_tile_id);
  if (CE_OK != e) { return CE_TRUNC_FLOOD; }
  
  printf("  + Flood tile is %u (0x%x).\n", flood_tile_id, flood_tile_id);

  e = room_flood_tile (flood_tile_id,
                       tiledata,
                       vram,
                       world->pixel_base_pairs,
                       world->animation_speeds);
  if (CE_OK != e) { return e; }
  
  // FEATURE TILE ID + HELPER: room[N+2], room[N+1]
  // room[N+1]
  
  if (CE_OK != room_data_read(roomd, &room_features)) {
    return CE_TRUNC_ROOM_FEATURES;
  }
  
  // OK. room features:
  // - bit 0: have first round of N3X decor
  // - bit 1: have item pad
  // - bit 2: have ropes or ladders
  // - bit 3: have stars
  // - bit 4: have door
  // - bit 5: have pillars
  // - ...
  
  printf ("  + Room features byte: 0x%x (", room_features);
  print_room_features (room_features);
  printf (")\n");
  
  num_tiles_round1_round2 = 0;
  
  if (room_features & 1) {
  
    //trace ("35b5 n3x setup");
    // room [N+2] -- number of decor round 1 tiles
    //num_tiles_round1_round2 = room.read(CE_TRUNC_FEATURE_TILE_ID);
    if (CE_OK != room_data_read (roomd, &num_tiles_round1_round2)) {
      return CE_TRUNC_FEATURE_TILE_ID;
    }
    
    round_1_num_tiles = num_tiles_round1_round2 & 7;

    if (0 != round_1_num_tiles) {
      // 35be ???
      printf("  + Decor round 1; %u tile%s chosen from pool of IDs:\n      ",
             round_1_num_tiles, ((round_1_num_tiles==1)?"":"s"));
             
      e = room_n3x (roomd,
                    tiledata,
                    &room_features,          // potentially modified by The Well
                    config->star_port_destroyed,
                    config->well_drained,
                    config->ice_crystal,
                    config->crystals,
                    config->lab_opened,
                    config->battlements_lift_on,
                    round_1_num_tiles,       // num tiles to blit
                    room_id,
                    world->pixel_base_pairs,
                    world->animation_speeds,
                    vram,
                    sm_tactile_x_out,
                    sm_tactile_y_out);
                    
      if (CE_OK != e) { return e; }

    }

  } else {
    //~ room_N_plus_2 = 0;
  }
  
  room_features >>= 1;

  // room[M]
  if (CE_OK != room_data_read(roomd, &b)) {
    return CE_TRUNC_ROOM_M;
  }
  
//printf("b=%x\n", b);
  
  num_rectangles = (b >> 3) & 0x1f; // actually t400_ptr_L
  num_triangles = b & 7;
  skip_trgl_stuff = (num_triangles == 0); //(num_triangles > 0) ? 0 : 1;
  
  water_X1 = -1;
  water_X2 = -1;
  water_level = -1;
  
  e = room_rectangles(roomd,
                      num_rectangles,
                      room_id,
                      &water_X1,    // out
                      &water_X2,    // out
                      &water_level, // out
                      config->star_port_destroyed,
                      config->well_drained,
                      vram);
  if (CE_OK != e) { return e; }
  
  for (i=0; i < MAX_TRIANGLES; i++) {
    triangle_init (triangles + i);
  }

  if ( ! skip_trgl_stuff ) {
    e = room_triangles (roomd,
                        config->star_port_destroyed,
                        &num_triangles, // modified
                        triangles);
    if (CE_OK != e) { return e; }

    propagate_triangle_colours_and_plot_modes (triangles, num_triangles);
    
    e = fake_draw_triangles(triangles, num_triangles, vram, config->triangle_outlines_only);
    
    if (CE_OK != e) { return e; }
    
    /*
    draw_triangles(triangles,
                   cvr,
                   cc.triangle_outlines_only,
                   cc.extended_screen);
    */
    
    // have triangle on cr
    
  }
  
  // Lskip_trngl_stuff
      
  item_pad_X = 0xff;
  item_pad_Y = 0xff;
  
  if (room_features & 1) {
    //~ trace2("3737 ROOM CONTAINS ITEM PAD");
    e = room_item_pad (roomd,
                       config->item_tile_id,
                       &item_pad_X, // out
                       &item_pad_Y, // out
                       BLIT_STA, //"STA", // ps.blit500_selfmod_op,
                       tiledata,
                       vram,
                       world->pixel_base_pairs,
                       world->animation_speeds);
    if (CE_OK != e) { return e; }
    printf("  + Room has an item pad at (%u, %u).\n", item_pad_X, item_pad_Y);
  }
  
  room_features >>= 1;

  // do the second layer of n3x decor
  
  //~ ps.set_blit500_selfmod_op_all("STA");
  
  round_2_num_tiles = (num_tiles_round1_round2 >> 3) & 0x1f;

  if ((num_tiles_round1_round2 & 0xf8) != 0) {       // 377f
  
    printf("  + Decor round 2; %u tile%s chosen from pool of IDs:\n      ",
           round_2_num_tiles, ((round_2_num_tiles==1)?"":"s"));
           
    e = room_n3x (roomd,
                  tiledata,
                  &room_features, // potentially modified by The Well
                  config->star_port_destroyed,
                  config->well_drained,
                  config->ice_crystal,
                  config->crystals,
                  config->lab_opened,
                  config->battlements_lift_on,
                  round_2_num_tiles, // tiles_to_blit
                  room_id,
                  world->pixel_base_pairs,
                  world->animation_speeds,
                  vram,
                  sm_tactile_x_out,
                  sm_tactile_y_out);
    if (CE_OK != e) { return e; }
    
  }
//~ ps.set_blit500_selfmod_op_all("EOR");

  if ( CE_OK != room_data_inbounds (roomd) ) {
    // terminate early
    return CE_OK;
  }
  
  have_ropes_ladders = room_features & 1;
  room_features >>= 1;
  
  // room features decides whether to run the next bit
  // if it's zero, we skip straight to the stars
  
  if (have_ropes_ladders) {
    
    // ROPES & LADDERS

    num_ropes = num_ladders = 0;
    e = get_num_ropes_and_ladders(roomd,
                                  &num_ropes,
                                  &num_ladders);
    if (CE_OK != e) { return e; }
    
    if ( CE_OK != room_data_inbounds (roomd) ) {
      return CE_OK; // terminate early
    }
    
    //print "ROPES: ".num_ropes."; LADDERS = ".num_ladders."\n";

    e = room_ropes (roomd, num_ropes, vram);
    if (CE_OK != e) { return e; }
    
    if (CE_OK != room_data_inbounds (roomd) ) {
      return CE_OK;  // terminate early
    }
    
    e = room_ladders(roomd,
                     num_ladders,
                     world->pixel_base_pairs,
                     vram);
    if (CE_OK != e) { return e; }

    if ( CE_OK != room_data_inbounds (roomd) ) {
      return CE_OK; // terminate early
    }
    
  }
  
  // S.T.A.R.S.
  
  if (room_features & 1) {
    e = room_stars(vram);
    if (CE_OK != e) { return e; }
  }
  
  room_features >>= 1;
  
  // FLASKS
  
  // does room have a flask?
  flask_id = -1;
  
  for (i=0; i < WORLD_NUM_FLASKS; i++) {
    if (world->flasks_room_ids[i] == room_id) {
      flask_id = i;
      break;
    }
  }
  
  if (flask_id >= 0) {
    e = room_flask (0xff & flask_id,
                    tiledata,
                    vram,
                    world->pixel_base_pairs,
                    world->animation_speeds,
                    world->flasks_positions,
                    &flask_x8,
                    &flask_y8);
    *flask_x_out = flask_x8;
    *flask_y_out = flask_y8;
    if (CE_OK != e) { return e; }
  }
  
  // DOORS
  
  if (room_features & 1) {
    // FIXME: in game, door status comes from t_track_doors[]
    // overridden here by doors_unlocked
    door_x_coord = -1;
    door_y_coord = -1;
    e = room_door(roomd,
                  tiledata,
                  vram,
                  world->pixel_base_pairs,
                  world->animation_speeds,
                  config->doors_unlocked,
                  &door_x_coord,
                  &door_y_coord);
  }
  
  room_features >>= 1;
  
  // PILLARS
  
  if (room_features & 1) {
    // 3923
    e = room_pillars (roomd,
                      tiledata,
                      world->pixel_base_pairs,
                      world->animation_speeds,
                      vram);
    if (CE_OK != e) { return e; }
  }
  
  room_features >>= 1;
  
  return CE_OK;
  
}



static citerr_t room_pillars  (room_data_t *room,
                               tile_data_t *tiledata,
                               u8_t *pixel_base_pairs,
                               u8_t *animation_speeds,
                               vram_t *vram) {

  // 3923
  
  s8_t tile_width_dummy;
  u8_t t_500[T_500_LEN];
  u8_t t_xyzzy[4];
  tile_slot_t tileslots[NUM_TILESLOTS];
  u8_t tile_slot_dummy;
  citerr_t e;
  u8_t b,c,p1,x,y,xy,i,carry_dummy;
  s8_t a;
  
  printf("  + Room has pillars: ");
 
  tile_width_dummy = 0;
  tile_slot_dummy = 0;
  
  tile_data_wipe400(tiledata);
  
  e = unpack_tile_gfx(tiledata,
                      TID_PILLAR_TOP,
                      t_500,
                      t_xyzzy,
                      &tile_slot_dummy,
                      0xff, 0xff, // t400 ptr L, H
                      0xff, 0xff, // tile_500buf_pos_1_initial, .._2_initial
                      &tile_width_dummy,
                      tileslots,
                      1, // initialise
                      pixel_base_pairs,
                      animation_speeds,
                      &carry_dummy);

  // ?? what's this byte? something applying to all pillars in room
  if (CE_OK != room_data_read(room, &p1)) {
    return CE_TRUNC_PILLARS_1;
  }
  
  a = p1 & 0xf;       // decor_bar
  b = p1 >> 2;        // tileptr_L_or_misc :: FIXME: possible carry issues?
  c = (p1 >> 3) & 6;  // tileptr_H
  
  // TODO: figure out what on earth is going on in here
  
  do { // 3939: P_pillars_1
  
    u16_t vram_ptr;
  
    // weird loop ...
    
    // only room #183 -- the Star Port teleporter -- runs this loop more than once
    // without the second iteration, it doesn't draw the right-hand pink pillar
    // underneath the teleport platform
    
    do {  // 3939: P_pillars_1
      if (CE_OK != room_data_read (room, &xy)) {
        return CE_TRUNC_PILLARS_2;
      }
      room_byte_to_coordinates(xy, &x, &y);
      if (y >= 0x40) { break; } // 393e: bcs L3948
      b = y;
      c = xy & 0xf;
    } while (1);
    
    // L3948
    
    x += c + 1; // + 1 is for carry
    
    for (i=0; i<=3; i++) { // 3954: P_pillars_2
      // interesting ... edit the tile slots so they unpack to different positions in t_500
      tileslots[i].offset_500buf = b & 0x30;
    }
    
    printf("(%u, %u)", x, y);

    e = tile_plot_specify_tile_slot(tiledata,
                                    tileslots + 0,
                                    t_500,
                                    BLIT_EOR,
                                    x,
                                    y,
                                    1,
                                    0,
                                    0xff,
                                    vram);
    if (CE_OK != e) { return e; }
    
    vram_ptr = 0; // scope

    do { // 395f: P_pillars_3
    
      u8_t px;
    
      vram_ptr = coords_to_vram_ptr(x + 1, y + 8);
      y += 4;
      
      // continue downwards until collision
      e = vram_read(vram, vram_ptr, &px);
      if (CE_OK != e) { return e; }
        
      if ( 0 == (px & 0x80) ) {
        // tile slot 2 => pillar shaft
        e = tile_plot_specify_tile_slot(tiledata,
                                        tileslots + 2,
                                        t_500,
                                        BLIT_EOR,
                                        x + 1,
                                        y,
                                        1,
                                        0,
                                        0xff,
                                        vram);
        if (CE_OK != e) { return e; }
      } else { // collided
        break;
      }
      
      // technically this could just be while (1) I think, but
      // this is the test that the adc at the end of tile_plot
      // would imply:
      
    } while ( ( vram_ptr & 0xff00 ) < 0x8000 );
    
    // tile slot 1 => pillar base
    tile_plot_specify_tile_slot(tiledata,
                                tileslots + 1,
                                t_500,
                                BLIT_EOR,
                                x,
                                y,
                                1,
                                0,
                                0xff,
                                vram);
    
    a--; // a.k.a. decor_bar
    
    if (a >= 0) {
      printf("; ");
    }
    
  } while (a >= 0);
  
  printf("\n");
  
  return CE_OK;
  
}



static citerr_t room_door  (room_data_t *room,
                            tile_data_t *tiledata,
                            vram_t *vram,
                            u8_t *pixel_base_pairs,
                            u8_t *animation_speeds,
                            u8_t unlocked_mode,
                            s16_t *door_x_coord_out,
                            s16_t *door_y_coord_out) {
                    
  u8_t door_id;
  u8_t dummy8;
  u8_t tile_id;
  s8_t tile_width_dummy;
  u8_t tile_slot;
  u8_t t_500[T_500_LEN];
  u8_t t_xyzzy[4];
  tile_slot_t tileslots[NUM_TILESLOTS];
  citerr_t e;
  u8_t i;
  u8_t xy, x, y;
  u8_t carry_dummy;
  
  if (CE_OK != room_data_read(room, &door_id)) {
    return CE_TRUNC_DOOR_ID;
  }
  
  printf("  + Room has a door; ID is %u.\n", door_id);;

  if (unlocked_mode == DOORS_OPENED) {
    // skip over the next room byte and return
    if (CE_OK != room_data_read(room, &dummy8)) {
      return CE_TRUNC_DOOR_SKIP_OPEN;
    }
  }
  
  tile_id = door_id | TID_FIRST_DOOR;
  
  tile_width_dummy = 0;
  tile_slot = 0;

  tile_data_wipe400 (tiledata);
  
  e = unpack_tile_gfx(tiledata,
                      tile_id,
                      t_500,
                      t_xyzzy,
                      &tile_slot,
                      0xff, 0xff,
                      0xff, 0xff, // tile_500buf_pos_1_initial, .._2_initial
                      &tile_width_dummy,
                      tileslots,
                      1, // initialise
                      pixel_base_pairs,
                      animation_speeds,
                      &carry_dummy);
  if (CE_OK != e) { return e; }
                  
  if (unlocked_mode == DOORS_LOCKED) {
    // lock door
    // MJ assumes the length of the door sprite in t_500 is 16,
    // but we just use the length of t_500, which is actually
    // not 16 bytes but rather 4 bytes;
    // (we called unpack_tile_gfx with initialise = 1,
    // which means we got a fresh t_500)
    for (i=0; i < T_500_LEN /* 0xf */; i++) {
      t_500[i] |= 0xc0; // door is made "yellow" (impermeable); both collision bits = 1
    }
  }
  // yank room coordinates from stream and draw it
  //xy = next_room_byte(buf, pos);
  
  if (CE_OK != room_data_read (room, &xy)) {
    return CE_TRUNC_DOOR_POS;
  }
  
  room_byte_to_coordinates(xy, &x, &y);
  
  e = tile_plot_specify_tile_slot  (tiledata,
                                    tileslots + 0,
                                    t_500,
                                    BLIT_EOR, // ?? STA works fine too, but I think it's technically EOR
                                    x,
                                    y,
                                    0,
                                    0,
                                    0,
                                    vram);
                                    
  return e;
  
}



static citerr_t room_stars (vram_t *vram) {

  // 3848: P_stars_3
  
  s16_t i;
  u8_t sm2a;
  
  // P_stars_3 at 3848 copies first three bytes from t_shifter
  // into t_stars, and then replaces those three values in
  // t_shifter with 0x20.
  
  // we just simulate that here:

  u8_t t_shifter[6] = { 0x20, 0x20, 0x20, 0, 0, 0 };
  // FIXME: values just dumped from memory, actual origin untraced ...
  u8_t t_stars[4]   = { 0xdc, 0x78, 0x78, 0 };

  printf("  + Room has stars.\n");
  
  // 3853
  sm2a = 0;
  
  for (i=255; i>=0; i--) { // 385a: P_stars_1
  
    u8_t carry, a, sm2b, x;
    u16_t vp;
    citerr_t e;
    
    carry = 0;
    a = do_shifter(t_shifter, &carry); // carry not actually needed
    // force address high into VRAM range:
    a &= 0x7f;
    a |= 0x40;
    if (a < 0x49) { // 385d
      // carry clear
      a += 9;
    }
    // L3867
    // FIXME: superfluous; sm4 and sm2 are the same
    // these are VRAM high bytes
    //sm4b = a & 0xff;
    sm2b = a & 0xff;
    a = do_shifter (t_shifter, &carry);
    a = add_with_carry (a, sm2a, &carry);
    sm2a = a & 0xff;
    //sm4a = a & 0xff;
    // FIXME: overflow shenanigans
    vp = to_16bit(sm2b, sm2a);
    //vp = to_16bit(sm4b, sm4a);
    // check is superfluous?
    if (vp < VRAM_GFX_WINDOW_START) {
      printf("ERROR: vp read < VRAM_GFX_WINDOW_START (%x)\n", vp);
      return CE_STARS_UNDERFLOW;
    }
    // 3879
    e = vram_read(vram, vp, &x);
    if (CE_OK != e) { return e; }
    //x = cvr.read(vp);
    if (x == 0) { // only plot a star if the read pixel is empty
      // ??? FIXME ??? STX foo here? doesn't actually do anything?
      // read by pillar code later ??? what
      // foo = x;
      if (0 == (a & 1)) {
        a = 0x15 & t_shifter[1];
      } else { // POSSIBLE FIXME; BPL at 3888
        a = 0x2a & t_shifter[1];
      }
      vp = to_16bit(sm2b, sm2a);
      e = vram_write(vram, vp, a);
      if (CE_OK != e) { return e; }
      //cvr.write(vp, a);
    }
  }
  
  // plotting done
  // copy t_stars back to t_shifter
  
  // 3895
  t_shifter[0] = t_stars[1];
  t_shifter[1] = t_stars[2];
  t_shifter[2] = t_stars[3];
  
  // FIXME: return t_shifter??
  
  return CE_OK;
  
}


static u8_t do_shifter (u8_t t_shifter_inout[6], u8_t *carry_out) {

  u8_t a, carry;
  s8_t j;
  
  a = t_shifter_inout[0];
  
  a &= 0x48;
  a += 0x38;
  
  // asl pair at 40b6
  carry = (a >> 6) & 1;

  for (j=2; j>=0; j--) {
    u16_t i;
    // ROLs at 40b8, 40ba, 40bc
    i = 0x1fe & (t_shifter_inout[j] << 1);
    t_shifter_inout[j] = (i & 0xfe) | carry; // carry into low bit
    carry = (i>>8) & 1; // carry out of high bit
  }
  
  *carry_out = carry;
  
  return t_shifter_inout[0];
  
}





citerr_t room_ladders (room_data_t *room, //char *s,
                       u8_t num_ladders,
                       u8_t *pixel_base_pairs,
                       vram_t *vram) {
                      
  u8_t d;
                      
  for (d=0; d < num_ladders; d++) {
  
    // 37dc
    
    u8_t q, b, b2;
    s8_t x, y, p;
    u8_t colour1, colour2;
    s32_t vp;
  
    // coordinate
    if (CE_OK != room_data_read (room, &b)) {
      return CE_TRUNC_LADDER_POS;
    }
  
    y = b & 0xf0;
    x = 10 * (b & 0xf);
    
    vp = coords_to_vram_ptr(x, y + 8); // FIXME: fix this up so it doesn't need ParserState
    
    if (CE_OK != room_data_read (room, &b2)) {
      return CE_TRUNC_LADDER_2;
    }
    
    // 7c is 01111100 so we're looking at five bits here, 0-32
    // this is the ladder length??
    p = (0x7c & b2) >> 2;
    
    // meanwhile the two low bits are used as the colour parameter
    q = (b2 & 3) << 1; // multiply by 2
    
    colour2 = pixel_base_pairs[8 + q];  // 37f6: P_ladder_get_colour
    colour1 = pixel_base_pairs[8 + q + 1];
    
    do { // 3802: P_ladder_outer
    
      s8_t n;
      citerr_t e;
      s8_t j, k;
    
      for (n = 0x1f; n >= 0; n--) { // 3812: P_ladder_blit_3
        // sanity
        if ((vp + n) > 0xffff) {
          //printf("ERROR: room_ladders(): (vp + n) > 0xffff (%d)\n", vp + n);
          //return CE_LADDERS_VP_OVERFLOW;
          continue;
          //return CE_OK;
        }
        if ((vp + n) < 0) {
          //printf("ERROR: room_ladders(): (vp + n) < 0 (%d)\n", vp + n);
          //return CE_LADDERS_VP_UNDERFLOW;
          continue;
          //return CE_OK;
        }
        e = vram_write (vram, 0xffff & (vp + n), 0x40);  // write 0x40; black, climbable (collision b01)
        if (CE_OK != e) { return e; }
      }
      
      // 3817
      e = draw_ladder_rung(0x1c, colour1, 0xffff & vp, vram);
      if (CE_OK != e) { return e; }
      
      if (p != 0) {
        // 3822
        e = draw_ladder_rung(0x18, colour1, 0xffff & vp, vram);
        if (CE_OK != e) { return e; }
      }
      
      // L3827
      
      // sanity
      if ((vp + 0x27) > 0xffff) {
        printf("ERROR: room_ladders(): vp + 0x27 > 0xffff\n");
        return CE_LADDERS_VP_OVERFLOW_2;
      }
 
      // right-hand rail
      for (j = 0x27, k = 0x7;
           k >= 0;
           j--, k--) {
        e = vram_write (vram, 0xffff & (vp + j), colour2);
        if (CE_OK != e) { return e; }
      }
      
      // left-hand rail
      for (j = 0x7; j >= 0; j--) {
        e = vram_write (vram, 0xffff & (vp + j), colour2);
        if (CE_OK != e) { return e; }
      }
      
      // in the asm this is right at the top of the loop, and just jumped over
      // on the first iteration
      
      //tmp = vp - 0x280;
      //if (tmp < 0) {
        //printf("ERROR: room_ladders(): vp - 0x280 < 0\n");
        //return CE_LADDERS_VP_UNDERFLOW;
        //break; // ???
      //}
      
      vp -= 0x280;
      
      p--; // 383a dec tileptr_L_or_misc
      
    } while (p >= 0);
    
  }
  
  return CE_OK;
  
}

// TODO: convert vram pointer to linear address,
// then just draw directly to the linear bitmap rather than MODE 2
static citerr_t draw_ladder_rung (s8_t y, u8_t c, u16_t vp, vram_t *vram) {
  u32_t tmp;
  citerr_t e;
  tmp = vp + y;
  if (tmp > 0xffff) {
    printf("ERROR: draw_ladder_rung(): vp overflow\n");
    return CE_LADDER_RUNG_VP_OVERFLOW;
  }
  for (; y >= 0; y -= 8) {
    e = vram_write (vram, vp + y, c);
    if (CE_OK != e) { return e; }
  }
  return CE_OK;
}



static citerr_t room_flask (u8_t flask_id,
                            tile_data_t *tiledata,
                            vram_t *vram,
                            u8_t pixel_base_pairs[WORLD_NUM_PXPAIRS],
                            u8_t animation_speeds[WORLD_NUM_ANIMSPEEDS],
                            u8_t flasks_positions[WORLD_NUM_FLASKS],
                            u8_t *flask_x_out,
                            u8_t *flask_y_out) {

  citerr_t e;
  u8_t flask_tile_id;
  s8_t tile_width_dummy;
  tile_slot_t tileslots[NUM_TILESLOTS];
  u8_t t_500[T_500_LEN];
  u8_t t_xyzzy[4];
  u8_t tile_slot;
  u8_t carry_dummy;
  u8_t xy;
  u8_t low_flask;
  
  low_flask = (flask_id < WORLD_NUM_LOW_FLASKS);

  flask_tile_id = low_flask ? TID_LOW_FLASK : TID_HIGH_FLASK;
  
  tile_width_dummy = 0;
  tile_slot = 0;
  carry_dummy = 0;
  
  e = unpack_tile_gfx(tiledata,
                      flask_tile_id,
                      t_500,   // out
                      t_xyzzy, // out
                      &tile_slot,
                      0, 4, // t_400 ptr L, H -- 0 and 4?
                      0, 0, // tile_500buf_pos_1_initial, .._2_initial
                      &tile_width_dummy,
                      tileslots, // out
                      1, // initialise
                      pixel_base_pairs,
                      animation_speeds,
                      &carry_dummy);
  if (CE_OK != e) { return e; }

  xy = flasks_positions[flask_id];
  room_byte_to_coordinates(xy, flask_x_out, flask_y_out);
  (*flask_x_out) += 2; // 38cf

  printf("  + Room has a flask (id %u, %s, coords [%u, %u]).\n",
         flask_id,
         (low_flask ? "green" : "red"),
         *flask_x_out,
         *flask_y_out);

  e = tile_plot_specify_tile_slot(tiledata,
                                  tileslots + 0, //(tile_slot - 1),
                                  t_500,
                                  BLIT_EOR,
                                  *flask_x_out,
                                  *flask_y_out,
                                  0,
                                  0,
                                  0,
                                  vram);
  
  return e;
  
}



static citerr_t room_ropes  (room_data_t *room,
                             u8_t num_ropes,
                             vram_t *vram) {
                    
  // 378c
  
  u8_t r;

  for (r=0; r < num_ropes; r++) {
  
    u8_t b, x, y;
    u16_t vp;
  
    if (CE_OK != room_data_read (room, &b)) {
      return CE_TRUNC_ROPE_1;
    }
    
    y = 0xf0 & b;
    x = 10 * (0xf & b);
    x += 4;
    if (y < 0xf0) {
      y |= 0x8;
    }
    
    vp = coords_to_vram_ptr (x, y);
    
    do { // 37a7: P_rope_outer
    
      u8_t double_break;
      s8_t ry;
    
      ry = 7;
      double_break = 0;
      
      do { // 37a9: P_rope_inner
        s32_t tmp;
        u8_t px;
        citerr_t e;
        tmp = vp + ry;
        if (tmp > 0xffff) {
          printf("ERROR: room_ropes: vram_ptr overflow\n");
          return CE_ROPES_VRAM_PTR_OVERFLOW;
        }
        e = vram_read (vram, vp + ry, &px);
        if (CE_OK != e) { return e; }
        px &= 0x3f;
        // terminate rope as soon as pixel read is not black
        if (px != 0) {
          double_break = 1;
          break;
        }
        e = vram_write (vram, vp + ry, 0x6c);
        if (CE_OK != e) { return e; }
        //tmp = ry - 2;
        //if (tmp < 0) {
        //  printf("ERROR: room_ropes: vram_ptr underflow\n");
        //  return CE_ROPES_VRAM_PTR_UNDERFLOW;
        //}
        ry--;
        e = vram_write(vram, vp + ry, 0x5c);
        if (CE_OK != e) { return e; }
        ry--;
        // L37cb
      } while (ry >= 0);
      
      if (double_break) { break; } // to next rope
      vp -= 0x280;
      
    } while (vp >= 0x4800); // VRAM low byte
    
  } // next rope
  
  return CE_OK;
  
}


  // for debug outline triangle mode only
  // black is not present; brown is first colour
static u8_t resistor_colour_code[9][3] = {
  { 0x5e, 0x3b,  0x2 }, // brown
  { 0xff,  0x0,  0x0 }, // red
  { 0xff, 0x6d,  0x0 }, // orange
  { 0xff, 0xff,  0x0 }, // yellow
  {  0x0, 0xff,  0x0 }, // green
  {  0x0,  0x0, 0xff }, // blue
  { 0x7e,  0x0, 0xcb }, // violet
  { 0x88, 0x88, 0x88 }, // grey
  { 0xff, 0xff, 0xff }  // white
};


static citerr_t fake_draw_triangles (triangle_t *tgls, s8_t nt, vram_t *vram, u8_t outlines_only) {

  s8_t i;
  citerr_t e, e2;
  u32_t num_spans[3];
  u32_t num_spans_alloced[3];
  raster_span_t *spans[3];
  //~ flood_pool_t floodpool;
  u8_t linebuf[SCRATCH_TRGL_BUF_HEIGHT * FRAMEBUF_WIDTH];
  s16_t /*left_start,*/ left_end, right_start /*, right_end*/;
  
  e2 = CE_OK;
  
  if ( nt == 0 ) { return CE_OK; }
  
  //~ memset(&floodpool, 0, sizeof(flood_pool_t));

//vram_init(vram, 176);
  
  printf("  + Room has %d triangle%s:\n", nt, ((nt==1)?"":"s"));
  
  for (i=0; i < nt; i++) {

    triangle_t *t;
    // u32_t span_ix;
    // s32_t p;
    u32_t j;
    u32_t count;
    u8_t r[2], g[2], b[2], c[2], pxc[2];

    s32_t x,y;
    u8_t run;
    
    e2 = CE_OK;
    
    memset(linebuf, 0, SCRATCH_TRGL_BUF_HEIGHT * FRAMEBUF_WIDTH);
  
    t = tgls + i;
    memset(spans, 0, sizeof(raster_span_t *) * 3);
    
    printf("      [%3d, %3d], [%3d, %3d], [%3d, %3d]; ",
           t->v1.x, t->v1.y, t->v2.x, t->v2.y, t->v3.x, t->v3.y);
    printf("colour 0x%x; plot mode 0x%x\n",
           t->fg_colour & 0xff, t->fg_plot_mode & 0xff);
          
    // everything is done in white at first
    //~ num_spans = 0;
    //~ num_spans_alloced = 0;
    
    memset(num_spans, 0, sizeof(u32_t) * 3);
    memset(num_spans_alloced, 0, sizeof(u32_t) * 3);
    
printf("FIXME: check for zero num_spans\n");
    
    e = bresenham (t->v1.x,
                   t->v1.y,
                   t->v2.x,
                   t->v2.y,
                   0,
                   linebuf,
                   spans + 0,
                   num_spans + 0,
                   num_spans_alloced + 0);
                   
    if (CE_OK != e) {
      if (spans[0] != NULL) {
        free(spans[0]);
      }
      //return e;
      e2 = e;
      break;
    }
//printf("total pixels = %u\n", sidelens[0]);
//~ u32_t j; printf("%s: ", spans[0].steep ? "steep" : "shallow"); for (j=0; j < num_spans; j++) { printf("(%d, %d) len %d, ", spans[j].start_x, spans[j].start_y, spans[j].len); } printf("\n");
//~ printf("num_spans = %u\n", num_spans);
    e = bresenham (t->v2.x,
                   t->v2.y,
                   t->v3.x,
                   t->v3.y,
                   0,
                   linebuf,
                   spans + 1,
                   num_spans + 1,
                   num_spans_alloced + 1);
    if (CE_OK != e) {
      if (spans[1] != NULL) {
        free(spans[1]);
      }
      free(spans[0]);
      //return e;
      e2 = e;
      break;
    }
//printf("total pixels = %u\n", sidelens[1]);
//~ printf("%s: ", spans[0].steep ? "steep" : "shallow"); for (j=0; j < num_spans; j++) { printf("(%d, %d) len %d, ", spans[j].start_x, spans[j].start_y, spans[j].len); } printf("\n");
//~ printf("num_spans = %u\n", num_spans);
    e = bresenham (t->v3.x,
                   t->v3.y,
                   t->v1.x,
                   t->v1.y,
                   0,
                   linebuf,
                   spans + 2,
                   num_spans + 2,
                   num_spans_alloced + 2);
    if (CE_OK != e) {
      if (spans[2] != NULL) {
        free(spans[2]);
      }
      free(spans[0]);
      free(spans[1]);
      //return e;
      e2 = e;
      break;
    }
//printf("total pixels = %u\n", sidelens[2]);
//~ printf("%s: ", spans[0].steep ? "steep" : "shallow"); for (j=0; j < num_spans; j++) { printf("(%d, %d) len %d, ", spans[j].start_x, spans[j].start_y, spans[j].len); } printf("\n");
//~ printf("num_spans = %u\n", num_spans);


      /*
      // fix spans; remove adjacent duplicates
      for (j=1; j < nspans; j++) {
        if (0 == memcmp(spans[0] + j - 1, spans[0] + j, sizeof(raster_span_t))) {
          // adjacent duplicate
printf("removed adjacent duplicate\n");
          // overwrite it
          memmove(spans[0] + j - 1, spans[0] + j, sizeof(raster_span_t) * (nspans - j));
          nspans--;
        }
      }
      */
      
    // scan the rendered lines
    for (y = 0; y < SCRATCH_TRGL_BUF_HEIGHT; y++) {
      for (x=0, run = 0, count = 0, left_end = -32768, right_start = -32768; // left_start = -32768, right_end = -32768
           x < FRAMEBUF_WIDTH;
           x++) {
        if (0 == run) {
          if (linebuf[x + (y * FRAMEBUF_WIDTH)] != 0) {
            run = 1;
            if (0 == count) {
              // left_start = x;
  //~ printf("left_start = %d\n", x);
            } else if (1 == count) {
              right_start = x;
  //~ printf("right_start = %d\n", x);
            }
            count++; // found another run
          }
        } else {
          if (linebuf[x + (y * FRAMEBUF_WIDTH)] == 0) {
            run = 0;
            if (1 == count) {
              left_end = x-1;
  //~ printf("left_end = %d\n", left_end);
            } else if (2 == count) {
              // right_end = x-1;
  //~ printf("right_end = %d\n", right_end);
            }
          }
        }
      } // next x
      if (count > 2) {
        // that's an error
        printf("ERROR: bresenham fail? detected >2 white runs on a rendered line (=%u)\n", count);
        e2 = CE_BUG;
        break;
      }
      
      if (count == 2) {
//printf("filling %d -> %d and %d -> %d\n", left_start, left_end, right_start, right_end);
        // this is the fill case
        for (x = left_end + 1; x < right_start; x++) {
          linebuf[x + (y * FRAMEBUF_WIDTH)] = 255;
        }
      }
    } // next y
    
 
    // now, copy from the scratch surface to the output one
    /*
    for (y = SCRATCH_TRGL_BUF_MARGIN / 2;
         y < (SCRATCH_TRGL_BUF_HEIGHT - SCRATCH_TRGL_BUF_MARGIN);
         y++) {
      for (x=0; x < FRAMEBUF_WIDTH; x++) {
        u8_t px;
        px = linebuf[x + (y * FRAMEBUF_WIDTH)];
        if (px == 255) {
          e = vram_linplot(vram, x, y - (SCRATCH_TRGL_BUF_MARGIN / 2), 255, 255, 255, 3); // fixme: rgbc
          if (CE_OK != e) {
            e2 = e;
            break;
          }
        }
      }
      if (CE_OK != e2) { break; }
    }
    */
    
    // determine colour pair
    

    if ( ! outlines_only ) {
      // real mode
      //~ $pxc = array();

      pxc[0]  = 0x55 & ((t->fg_colour) >> 1);
      pxc[1]  = 0x55 & t->fg_colour;
      
      for (j=0; j < 2; j++) { // left, right
        r[j] = (pxc[j] & 1)  ? 255 : 0;
        g[j] = (pxc[j] & 4)  ? 255 : 0;
        b[j] = (pxc[j] & 16) ? 255 : 0;
        c[j] = (pxc[j] & 64) ? 255 : 0;
      }

    } else {
      // debug mode -- outlines only
      // FIXME
      r[0] = r[1] = resistor_colour_code[i%9][0];
      g[0] = g[1] = resistor_colour_code[i%9][1];
      b[0] = b[1] = resistor_colour_code[i%9][2];
      c[0] = c[1] = 0; // ??
    }
    
    for (y = SCRATCH_TRGL_BUF_MARGIN / 2;
         y <= (SCRATCH_TRGL_BUF_HEIGHT - SCRATCH_TRGL_BUF_MARGIN);
         y++) {
      for (x=0; x < FRAMEBUF_WIDTH; x++) {
        u8_t px;
        u8_t xr, xg, xb, cr, cg, cb;
        s32_t y2;
        px = linebuf[x + (y * FRAMEBUF_WIDTH)];
        y2 = y - SCRATCH_TRGL_BUF_MARGIN / 2;
        // copy onto the surrogate MODE 2 bitmap
        //~ if ($ca["alpha"] == 0) {
        if (px != 0) {
          //$cr=$cg=$cb=-1;
          //CitadelVram::map_collision_colour(3 & ($t->fg_colour >> 6), $cr, $cg, $cb);
          map_collision_colour (3 & (t->fg_colour >> 6), &cr, &cg, &cb);
          // TODO: AND, OR modes unimplemented (but never used by Citadel)
          if (t->fg_plot_mode == 0) {
            // GCOL 0
            vram->linvis[     4*(x+(y2*FRAMEBUF_WIDTH)) ] = r[x&1];
            vram->linvis[1 + (4*(x+(y2*FRAMEBUF_WIDTH)))] = g[x&1];
            vram->linvis[2 + (4*(x+(y2*FRAMEBUF_WIDTH)))] = b[x&1];
            // collision
            vram->lincol[     4*(x+(y2*FRAMEBUF_WIDTH)) ] = cr;
            vram->lincol[1 + (4*(x+(y2*FRAMEBUF_WIDTH)))] = cg;
            vram->lincol[2 + (4*(x+(y2*FRAMEBUF_WIDTH)))] = cb;
          } else {
            // GCOL 3 (XOR)
            // dumb
            xr = vram->linvis[     4*(x+(y2*FRAMEBUF_WIDTH)) ] ? 255 : 0; //^= chr($r[$x&1]);
            xg = vram->linvis[1 + (4*(x+(y2*FRAMEBUF_WIDTH)))] ? 255 : 0; //^= chr($g[$x&1]);
            xb = vram->linvis[2 + (4*(x+(y2*FRAMEBUF_WIDTH)))] ? 255 : 0; //^= chr($b[$x&1]);
            xr ^= r[x&1];
            xg ^= g[x&1];
            xb ^= b[x&1];
            vram->linvis[     4*(x+(y2*FRAMEBUF_WIDTH)) ] = xr;
            vram->linvis[1 + (4*(x+(y2*FRAMEBUF_WIDTH)))] = xg;
            vram->linvis[2 + (4*(x+(y2*FRAMEBUF_WIDTH)))] = xb;
            
            xr = vram->lincol[     4*(x+(y2*FRAMEBUF_WIDTH)) ] ? 255 : 0; //^= chr($r[$x&1]);
            xg = vram->lincol[1 + (4*(x+(y2*FRAMEBUF_WIDTH)))] ? 255 : 0; //^= chr($g[$x&1]);
            xb = vram->lincol[2 + (4*(x+(y2*FRAMEBUF_WIDTH)))] ? 255 : 0; //^= chr($b[$x&1]);
            xr ^= cr;
            xg ^= cg;
            xb ^= cb;
            vram->lincol[     4*(x+(y2*FRAMEBUF_WIDTH)) ] = xr;
            vram->lincol[1 + (4*(x+(y2*FRAMEBUF_WIDTH)))] = xg;
            vram->lincol[2 + (4*(x+(y2*FRAMEBUF_WIDTH)))] = xb;
            
          }
          vram->linvis[3 + (4*(x+(y2*FRAMEBUF_WIDTH)))] = 0; //chr(0);
          vram->lincol[3 + (4*(x+(y2*FRAMEBUF_WIDTH)))] = 0; //chr(0);
        }
      }
    }

  }
  
  return e2;
  
}



static void map_collision_colour (u8_t col_colour, u8_t *r_out, u8_t *g_out, u8_t *b_out) {
  switch (col_colour) {
    case 1:  *r_out = 255; *g_out = *b_out = 0; break;
    case 2:  *r_out = *b_out = 0; *g_out = 255; break;
    case 3:  *r_out = *g_out = 255; *b_out = 0; break; // does this ever happen?
    default: *r_out = *g_out = *b_out = 0; break;
  }
}





  
static citerr_t get_num_ropes_and_ladders (room_data_t *room,
                                           u8_t *num_ropes_out,
                                           u8_t *num_ladders_out) {
  u8_t b;
  // 3789
  if (CE_OK != room_data_read (room, &b)) {
    return CE_GET_NUM_ROPES_LADDERS;
  }
  *num_ropes_out = 0xf & b;
  *num_ladders_out = 0xf & (b >> 4);
  return CE_OK;
}



static citerr_t room_flood_tile  (u8_t tile_id,
                                  tile_data_t *tiledata,
                                  vram_t *vram,
                                  u8_t pixel_base_pairs[WORLD_NUM_PXPAIRS],
                                  u8_t animation_speeds[WORLD_NUM_ANIMSPEEDS]) {
                                  
  s8_t tile_width_maybe;
  tile_slot_t tileslots[NUM_TILESLOTS];
  u8_t next_tile_slot_dummy;
  u8_t t_500[T_500_LEN];
  u8_t t_xyzzy[4];
  u8_t carry;
  u8_t xcoord, ycoord;
  u16_t vram_ptr, vram_ptr_2;
  citerr_t e;
  u16_t i;
  u32_t vp_sanity;
  
  for (i=0; i < NUM_TILESLOTS; i++) {
    tile_slot_init(tileslots + i);
  }
  
  // FLOOD TILE ID: room[N]
  
  //trace (sprintf("352c room_N_flood_tile (0x%x)", tile_id));
  
  tile_width_maybe = -1;
  next_tile_slot_dummy = 0;
  
  tile_data_wipe400(tiledata);
 
  e = unpack_tile_gfx  (tiledata,
                        tile_id,
                        t_500,
                        t_xyzzy,
                        &next_tile_slot_dummy,
                        -1, -1,
                        -1, -1, // tile_500buf_pos_1_initial, .._2_initial
                        &tile_width_maybe,
                        tileslots,
                        1,
                        pixel_base_pairs,
                        animation_speeds,
                        &carry);
  if (CE_OK != e) { return e; }
  
//printf("tile_width_maybe = %x\n", tile_width_maybe);
//exit(0);
  
  tiledata->ptr_L = tileslots[0].vram_target | 4; // t_tslots_* parallel arrays always use [0] as the flood fill tile slot?
  ycoord = 0x48;
  xcoord = 0;
  
  // place first copy of tile in top-left of screen
  
  e = tile_plot_a_slot_0 (tiledata,
                          tileslots + 0,
                          t_500,
                          BLIT_STA, // ps.blit500_selfmod_op,
                          xcoord,
                          ycoord,
                          vram); // slot 0, force X-coord=0
  if (CE_OK != e) { return e; }
  
  // derive initial VRAM SOURCE (??) from foobar
  
  vram_ptr_2 = coords_to_vram_ptr(xcoord, ycoord); // compute misc16 first, then ...
  
  // this works well enough to replace the strange
  // multiply loop at 3558
  ycoord += (4 * (1 + tileslots[0].height));
 
  //trace(sprintf("3548")); // tileptr=0x%x", ps.get_tileptr()));
  
  // derive initial VRAM TARGET (??) from foobar??
  vram_ptr = coords_to_vram_ptr(xcoord, ycoord); // 3548

//printf("lolo");
  // *** P_tiles_flood TRIPLE LOOP: &3562 - &35A6 ***
  // FIXME: split into function
  
  // misc16 points to TARGET tile position??

  do { // P_flood_outer_A
  
    s16_t y;
    
    u8_t start_vram_ptr_L;
    u8_t start_vram_ptr_H;
    s8_t width_accumulator; //start_vram_ptr_X;
  
    // misc16 values confirmed OK here ...
    
//printf("P_flood_outer_A: ptr = %x\n",tilesrc_inout.getptr());
  
    // first row?
    //y = tiledata->ptr_L; //tileptr_L;
    
    // P_flood_outer_init @ &3564:
    // FIXME: replace with for loop
    
    //do {
    for (y = tiledata->ptr_L;
         y >= 0;
         y--) {

      // sanity
      if (    (vram_ptr_2 + y < VRAM_GFX_WINDOW_START)
           || (vram_ptr_2 + y >= (VRAM_GFX_WINDOW_START + VRAM_LENGTH))) {
        printf("ERROR: P_flood_outer_init: vram_ptr_2 (0x%x) + y (0x%x) over/under VRAM "
               "(START = 0x%x, LEN = 0x%x)\n",
               vram_ptr_2, y, VRAM_GFX_WINDOW_START, VRAM_LENGTH);
        return CE_FLOOD_OUTER_INIT;
      }
     
      e = vram_copy2px (vram, vram_ptr + y, vram_ptr_2 + y);
      if (CE_OK != e) { return e; }
      //y--;
      
    }// while (y >= 0);
    
    // 356b - 3575; set up t_xyzzy with copy of misc16
    // note that this means t_xyzzy is now also infected
    // with the VRAM offset disease ...
    
    start_vram_ptr_L = 0xff & vram_ptr_2;
    start_vram_ptr_H = 0xff & (vram_ptr_2 >> 8);
    
    //width_accumulator = 0;
    
    //t_xyzzy[0] = 0xff &  vram_ptr_2;
    //t_xyzzy[1] = 0xff & (vram_ptr_2 >> 8);
    //t_xyzzy[2] = 0; // FIXME: use a local instead?
    
//printf("ptr_L = %x\n", tiledata->ptr_L);
    
    // P_flood_inner @ 3577
    //for (width_accumulator = 0;
    //     width_accumulator < 0x50;
    //     width_accumulator += tile_width_maybe) {
    
    width_accumulator = 0;
    
    do {
      
      vram_ptr_2 += tiledata->ptr_L + 1;   // 3577 to 3582
      
      width_accumulator += tile_width_maybe;
      
      if (width_accumulator >= 0x50) {
        break; // BCS &3598 at 358B
      }
      
      //y = tiledata->ptr_L; //tileptr_L; // 358d
      
      // NOTE that tileptr_L is apparently never updated
      // for the whole of this process, so it's effectively
      // an argument passed in to this process (at least for Pyramid)
      
      //printf("358d tileptr_L = %x\n", tiledata->ptr_L);
      //printf("width_acc = %x\n", width_accumulator);
      
      // P_flood_hard @ 358f
      // FIXME: replace with for loop?
      //do {
      for (y = tiledata->ptr_L; y >= 0; y--) {
      
        u16_t vram_p;
        
//printf("xyzzy[1] = %x, xyzzy[0] = %x\n", t_xyzzy[1], t_xyzzy[0]); exit(0);
      
        vram_p = to_16bit (start_vram_ptr_H, start_vram_ptr_L); //t_xyzzy[1], t_xyzzy[0]);
        
        // skip ROM overwrites
#warning FIXME: check is too strict, extended height mode will fail
        if ((vram_ptr_2 + y) < VRAM_GFX_WINDOW_START + VRAM_LENGTH) {
          e = vram_copy2px (vram, vram_ptr_2 + y, vram_p + y);
          if (CE_OK != e) { return e; }
        }
        
        //y--;
        
      } // while (y >= 0);
    
    } while ( 1 );

    vp_sanity = vram_ptr + 0x280;
    
    // more sanity
//    if (    (vp_sanity < VRAM_GFX_WINDOW_START)
//         || (vp_sanity >= (VRAM_GFX_WINDOW_START + VRAM_LENGTH))) {
    if (vp_sanity > 0xffff) {
      printf("ERROR: flood: vram_ptr (0x%x) would exceed ffff\n", vp_sanity);
      return CE_FLOOD_VRAM_PTR_SANITY;
    }
    
    vram_ptr = 0xffff & vp_sanity; //0x280;
    
    // hacked; condition in 6502 (BPL @ &35a6) is >=0,
    // but this is unsigned so < 0x80 will do instead
    
//printf("flood: dest = %x, src = %x\n", vram_ptr, vram_ptr_2);

  } while (vram_ptr_2 < 0x8000);
  
  return CE_OK;

}

u16_t coords_to_vram_ptr (u8_t x, u8_t y) {
  u8_t carry;
  return coords_to_vram_ptr_with_carry(x, y, &carry);
}


u16_t coords_to_vram_ptr_with_carry (u8_t _x, u8_t _y, u8_t *carry_out) {

  u16_t vram_ptr_L, vram_ptr_H;
  u16_t x, y;
  u8_t yreg, carry, a;
  u16_t ptr_out;

  vram_ptr_L = 0;
  vram_ptr_H = 0;
  x = _x;
  y = _y;

  // I think it does the X-component first:
  
  vram_ptr_L = (x << 2) & 0xf8; // vram ptr L starts out as X-coord * 4
  
  yreg = (0x80 & x) ? 0x34 : 0x32;
  if (x & 0x40) {
    yreg++; // at this point yreg may be x32, x33, x34 or x35
  }
  
  // and then the Y-component:
  a = (((y & 0xf8) * 5) / 8);
  vram_ptr_H = a / 2; // vram ptr H starts out as 5/16 of the X-coordinate
  
  carry = a & 1;
  a = y & 4;
  if (carry) {
    //trace("252d coords_to_vram_ptr");
    a = add_with_carry (a, 0x7f, &carry); // modifies carry
  }
  a = add_with_carry (a, vram_ptr_L, &carry);
  if (carry) {
    vram_ptr_H++; // implement carry from low byte to high byte
    carry=0;
  }
  a = add_with_carry (a, 0x80, &carry);
  vram_ptr_L = a; // wtf
  // high byte = high byte + (0x32 to 0x35) + carry
  a = add_with_carry(yreg, vram_ptr_H, &carry);
  vram_ptr_H = a;
  if (a < VRAM_START_HIGH_BYTE) {
    vram_ptr_H = VRAM_START_HIGH_BYTE;
  }
      
  *carry_out = carry;
  ptr_out = to_16bit(vram_ptr_H, vram_ptr_L);
  
  return ptr_out;
  
}


u8_t add_with_carry (u8_t a, u8_t b, u8_t *carry_inout) {
  u16_t x;
  x = (0xff & a) + (0xff & b) + (*carry_inout ? 1 : 0);
  *carry_inout = 0;
  if (x > 0xff) {
    *carry_inout = 1;
  }
  return (x & 0xff);
}

// OK. room features:
// - bit 0: have first round of N3X decor
// - bit 1: have item pad
// - bit 2: have ropes or ladders
// - bit 3: have stars
// - bit 4: have door
// - bit 5: have pillars
static void print_room_features (u8_t b) {
  u8_t i;
  u8_t need_comma;
  need_comma = 0;
  for (i=0; i < 6; i++, b >>= 1) {
    if (b & 1) {
      if (need_comma) { printf(", "); }
      if (i==0) { printf("decor1");  } // b & 0x01
      if (i==1) { printf("pad");     } // b & 0x02
      if (i==2) { printf("rop/lad"); } // b & 0x04
      if (i==3) { printf("stars");   } // b & 0x08
      if (i==4) { printf("door");    } // b & 0x10
      if (i==5) { printf("pillars"); } // b & 0x20
      need_comma = 1;
    }
  }
}


// RENAME to plot_tile_decorations() or similar
static citerr_t room_n3x  (room_data_t *room,
                           tile_data_t *tiledata,
                           u8_t *room_features_inout,
                           u8_t star_port_destroyed,
                           u8_t well_drained,
                           u8_t ice_crystal,
                           u8_t crystals[NUM_CRYSTALS],
                           u8_t lab_opened,
                           u8_t battlements_lift_on,
                           u8_t tiles_to_blit,
                           u8_t room_id,
                           u8_t pixel_base_pairs[WORLD_NUM_PXPAIRS],
                           u8_t animation_speeds[WORLD_NUM_ANIMSPEEDS],
                           vram_t *vram,
                           s16_t *sm_tactile_x_out,
                           s16_t *sm_tactile_y_out) {
                     
  //trace("32f4 room_n3x");
  
  // u8_t tile_attributes;
  u8_t decor_foo;
  u8_t sm_tile_slot;
  u8_t carry;
  u8_t selfmod_lda;
  u8_t t_500[T_500_LEN];
  u8_t t_xyzzy_dummy[4];
  u8_t tile_slot_dummy;
  tile_slot_t tileslots[NUM_TILESLOTS];
  
//return CE_OK;
  
  //printf ("room_n3x (features=0x%x, tiles_to_blit=%u)\n", room_features_inout, tiles_to_blit);
  // tile_attributes = 0;
  decor_foo = 0;
  
  sm_tile_slot = 0xff;
  
  carry = 0; // FIXME FIXME
  selfmod_lda = 0xff;
  
  //t_500 = array();
  //t_xyzzy_dummy = array();
  
  memset(t_500, 0, T_500_LEN);
  memset(t_xyzzy_dummy, 0, 4);
  memset(tileslots, 0, sizeof(tile_slot_t) * NUM_TILESLOTS);
  
  tile_slot_dummy = 0; //0xff;
  
  tile_data_wipe400(tiledata);
  tile_data_setptr(tiledata, 0xffff);
      
  while ((tiles_to_blit > 0) && (CE_OK == room_data_inbounds (room))) { // P_decor_2
  
    u8_t skip_switches_and_room_modifications;
    u8_t rb, rb2;
    u8_t r0_low_nyb, r0_high_nyb;
    u8_t lab_modification_done;
    s8_t decor_bar;
    u8_t recompute_tileptr_next_n3x;
    u8_t tile_id;
    u8_t dummy;
    s8_t tile_width_maybe_dummy;
    citerr_t e;
    u8_t xreg;
    // u16_t tp16_1, tp16_2;
    u8_t ycoord, xcoord;
    u8_t tx, ty;
    u8_t saved_xcoord;
  
    //trace3(sprintf("P_decor_2: %x\n", tld.getptr()));

    skip_switches_and_room_modifications = 0;
    
    // overflow should be impossible here
    if (CE_OK != room_data_read(room, &rb)) {
      return CE_TRUNC_N3X_1;
    }

    //trace2(sprintf("32fd room_n3x read room[N+3+X] = [0x%x | 0x%x]", (rb >> 4) & 0xf, rb & 0xf));
    //printf ("room[N3X] at 0x%x is 0x%x\n", i, rb);
    //printf ("tileptr_H=0x%x, tileptr_L=0x%x\n", ps_inout.tileptr_H, ps_inout.tileptr_L);

    r0_low_nyb = rb & 0xf;
    r0_high_nyb = (rb >> 4) & 0xf;
    
    if (r0_high_nyb != 0) {
      //trace("3300");
      if (r0_high_nyb >= 4) { // rdata byte was 0x4x
        //trace("3302");
        skip_switches_and_room_modifications = 1;
        carry = 1; // bcs Ldecor_1
      } else if (r0_high_nyb == 3) { // rdata byte was 0x3x
        //trace("330c");
        sm_tile_slot = r0_low_nyb & 0xf;
        //printf ("setting sm_tile_slot = 0x%x [a]\n", ps_inout.sm_tile_slot);
        //trace3(sprintf("before 330c continue: %x\n", tld.getptr()));
        continue; // next N3X
      }
      
      if ( ! skip_switches_and_room_modifications ) { // code at 3311
        //trace("3311");
        // check coords_high bit 1
        selfmod_lda = ((r0_high_nyb << 4) & 0x20);
        //printf ("selfmod_lda = 0x%x\n", ps_inout.selfmod_lda);
        decor_foo = r0_low_nyb & 0xf;
        //printf ("decor_foo = 0x%x\n", ps_inout.decor_foo);
        //if (0) { // ?? BPL at 331c
        //  skip_switches_and_room_modifications = 0; // fall through
        //} else {
        //trace3(sprintf("before 3311 continue: %x\n", tld.getptr()));
        continue; // next N3X
      }
    }
    
    lab_modification_done = 0;
    decor_bar = 0xff;
    
    //trace3(sprintf("before room mods: %x\n", tld.getptr()));
    
    if ( ! skip_switches_and_room_modifications ) { // redundant ??? see BPL comment above
    
      //trace("331e room_n3x Lswitch_stuff");
      //print "+ switch stuff: pos="; printf("0x%x", i); print "\n";
      decor_bar = r0_low_nyb & 0xf;
      if (CE_OK != room_data_read(room, &rb2)) {
        return CE_TRUNC_N3X_2;
      }
//      rb2 = room.read(CE_TRUNC_N3X_2);  // read room[N3Xi]
      //trace(sprintf("3325 room_n3x read room[N+3+X+i]=0x%x", rb2));
      sm_tile_slot = (rb2 & 1) ? 1 : 0;
      //printf ("setting sm_tile_slot = 0x%x [b]\n", ps_inout.sm_tile_slot);
      //~ cr_inout.switch_stuff[] = (rb2 >> 1) & 0x7f;
      carry = rb2 & 1;
      
      recompute_tileptr_next_n3x = 0;
      
      tile_id = ((rb2 >> 1) & 0x7f);

      //trace("3384 Linit_switches");
      
      if (tile_id == TID_WATER_SURFACE) {
        //trace("3388");
        if ((room_id == RID_WELL) && well_drained) {
          //trace("338e");
          // The Well: alternative layer, advance data stream by two bytes
          if (CE_OK != room_data_read(room, &dummy)) { return CE_TRUNC_N3X_WELL_SKIP; }
          if (CE_OK != room_data_read(room, &dummy)) { return CE_TRUNC_N3X_WELL_SKIP; }
          *room_features_inout = 1; // 339d
          recompute_tileptr_next_n3x = 1; // simulate RTS + fallthrough to unpack_tile_gfx() control flow
        } else {
          //trace("33a0"); // Lnot_the_well
          if (ice_crystal) {
            //trace("33a9 frozen water");
            tile_id++; // use frozen water surface instead
          }
        }
      }
      
      if (recompute_tileptr_next_n3x) { // skipped for 33a0 branch

        // simulate init_swchs_return RTS at 339f, return to room_n3x at 3330 and:
        //trace("3330");
        
        //printf("Well hack: tile %x\n", tile_id);
        
        // for the Well, this should be tile 5b, the Well green/cyan/black brickwork:
        // FIXME: for some reason it seems to be 0xc, the water surface, instead ...
        // FIXME: do we even need to do any of this nonsense before the break?
        
        tile_width_maybe_dummy = 0;
        printf("%u ", tile_id);
        e = unpack_tile_gfx(tiledata,
                            tile_id,
                            t_500,
                            t_xyzzy_dummy,
                            &tile_slot_dummy,
                            -1, -1,
                            -1, -1, // tile_500buf_pos_1_initial, .._2_initial
                            &tile_width_maybe_dummy,
                            tileslots,
                            1, // initialise
                            pixel_base_pairs,
                            animation_speeds,
                            &carry);
        if (CE_OK != e) { return e; }
        
        tiledata->ptr_L = (decor_bar << 1) & 6;
        tiledata->ptr_H = (decor_bar & 0xc);
        
        //trace3(sprintf("pre-break %x\n", tld.getptr()));
        
        // shrug. something to do with the double PLA -- it seems to work
        break;
        
      }
      
      xreg = tile_id;
    
      // 33aa
      //trace("33aa tile_id = 0x".sprintf("%x", xreg));
      if (tile_id == TID_SWITCH) {
        if (room_id == RID_48) { // x48
          //trace("33b4");
          *sm_tactile_x_out = 0x50;
          *sm_tactile_y_out = 0xe0;
          if (battlements_lift_on) { // switch flipped?
            sm_tile_slot++;
          }
        } else if (room_id == RID_WELL_WHEEL) { // x71
          //trace("33c3");
          *sm_tactile_x_out = 0x32;
          *sm_tactile_y_out = 0x70;
          //xreg = 0x32; <- NO -- it's set back to TID_SWITCH by LDX at 33e1
          if (well_drained) {
            sm_tile_slot++; // flip switch
          }
        } else if (room_id == RID_WEST_WING_SWITCH_MONK) { // x80
          //trace("33d2");
          *sm_tactile_x_out = 0xa;
          *sm_tactile_y_out = 0xd0;
        } else {
          //trace("33e1");
        }
       
      } else {
        //trace("33ac JMP L33e3");
      }
      
      //trace(sprintf("33e3 xreg=%x", xreg));
      
      // in certain situations, force tile_id = 0x1a, which is a blank tile

      //~ if ((xreg >= TID_LAST_CRYSTAL /* 5 */) || crystals[xreg]) { //(ps_inout.t_puzzle_flags[2 + xreg] != 0)) {
      // 33ec
      // interesting hack; there are two teleporters -- one on the local
      // planet, and one on the alien planet (rooms 19 and 183).
      // Room 183 has a hole in the floor and a rope leading downwards;
      // Room 19 has neither of these things.
      // This code modifies the room as a special case, in 6502 code.
      
      if ((xreg == TID_ROPE_BRACKET/*0x25*/) && (room_id == RID_19_TELEPORTER)) {
        //trace("3403 [z]");
        // get rid of the rope bracket for room 19
        xreg = TID_BLANK; //0x1a;
      } else if (star_port_destroyed
                 && ((xreg == TID_ZIGZAG/*0x26*/) || (xreg == TID_PARABOLIC_ANTENNA /*0x10*/))) {
        //trace("3403 [y]");
        // star port destroyed =>
        // get rid of the squiggle in the teleporter room,
        // and also the antenna on the Star Port parabolic dish
        xreg = TID_BLANK; //0x1a;
      }
      
      // for Sanctuary; blank out crystals that have
      // not been collected yet
      if ((xreg < 5) && crystals[xreg] == 0) {
        // 33ea; branch to 3403
        //trace("3403 [x]");
        xreg = TID_BLANK;
      }

      // 3405/3409

      // The Lab entrance wall has TID_BRICK_YELLOW_RED_MAGENTA ...
      // is this the code which punches a hole through The Lab wall?

      if ((xreg != TID_BRICK_YELLOW_RED_MAGENTA /*0x58*/) || ! lab_opened) {
        // simulated RTS code from earlier ...
        // simulate init_swchs_return RTS at 339f, return to room_n3x at 3330 and:
        //trace("3330");
        tile_width_maybe_dummy = 0;
        //print xreg." ";
        printf("%u ", xreg);
        e = unpack_tile_gfx  (tiledata,
                              xreg,
                              t_500,
                              t_xyzzy_dummy,
                              &tile_slot_dummy,
                              -1, -1,
                              -1, -1, // tile_500buf_pos_1_initial, .._2_initial
                              &tile_width_maybe_dummy,
                              tileslots,
                              1, // initialise
                              pixel_base_pairs,
                              animation_speeds,
                              &carry);
        if (CE_OK != e) { return e; }

        /*tp16_1 =*/ tile_data_getptr(tiledata);
        
        //trace3(sprintf("tp16_1 = %x\n", tp16_1));
        
        tiledata->ptr_L = (decor_bar << 1) & 6; // values saved from before?
        tiledata->ptr_H = (decor_bar & 0xc);
        
        /*tp16_2 =*/ tile_data_getptr(tiledata);
        
        //trace3(sprintf("tp16_2 = %x\n", tp16_2));
        
        // this continue will fire
        // for every room that gets to this point
        // *except* The Lab
        
        continue; // next N3X (P_decor_2) -- POSSIBLE FALLTHROUGH TO Ldecor_1 ???
        
      }
      
      // This piece of code is only run for an opened-up The Lab:
      
      // 3410
      //trace("3410");
      tile_width_maybe_dummy = 0;
      printf("%u ", xreg);
      e = unpack_tile_gfx (tiledata,
                            xreg,
                            t_500,
                            t_xyzzy_dummy,
                            &tile_slot_dummy,
                            -1, -1,
                            -1, -1, // tile_500buf_pos_1_initial, .._2_initial
                            &tile_width_maybe_dummy,
                            tileslots,
                            1, // initialise
                            pixel_base_pairs,
                            animation_speeds,
                            &carry);
      if (CE_OK != e) { return e; }
                                
      // skip one room byte; simulated JSR next_room_byte at 3413
      //room.read(CE_TRUNC_LAB_SKIP);
      if (CE_OK != room_data_read(room, &dummy)) {
        return CE_TRUNC_LAB_SKIP;
      }
      
      // reset tileptr to 0
      
      tile_data_setptr(tiledata, 0);
      //tld.setptr(0);
      
      lab_modification_done = 1;

    } // original code jumps back to Ldecor_1 (3420), we just fall through to it
    
    // there are three ways the original code gets to Ldecor_1:
    // - by jumping there from the top of room_n3x (achieved here via skip_switches_and_room_modifications)
    // - PROBABLY IMPOSSIBLE *** by falling through from Lswitch_stuff section;
    // - by jumping there from the very bottom of room_n3x, after the code directly above (here we just fall through to it)
    //   - this is The Lab modification
    
    //trace(sprintf("3342 Ldecor_1 ptr=%x", tld.getptr()));
    
    // Ldecor_1

    decor_bar = decor_foo;
    
    if ( ! lab_modification_done ) {
      room_byte_to_coordinates(rb, &tx, &ty);
      ycoord = add_with_carry (tiledata->ptr_H, ty, &carry);
      xcoord = add_with_carry (tiledata->ptr_L, tx, &carry);
    } else {
      // suspect this is the brick in The Lab which is knocked out
      // of the wall
      ycoord = 0xd0;
      xcoord = 0x30;
    }
    
    saved_xcoord = 0xff;
    
    // sanity
    if (0xff == sm_tile_slot) {
      printf("\nERROR: SANITY: 0xff == sm_tile_slot\n");
      return CE_NEGATIVE_SM_TILE_SLOT;
    }
    
    //trace3(sprintf("pre-do %x\n", tile_data_getptr(tiledata));
    
    do { // P_decor_3
    
      u8_t w;
    
      //trace3(sprintf("P_decor_3 %x\n", tld.getptr()));
      
      // sm_tile_slot contains tile slot?
      e = tile_plot_specify_tile_slot (tiledata,
                                       tileslots + sm_tile_slot,
                                       t_500,
                                       BLIT_STA, //ps_inout.blit500_selfmod_op,
                                       xcoord,
                                       ycoord,
                                       0,
                                       0,
                                       0,
                                       vram);
      if (CE_OK != e) { return e; }
                                   
      //trace3(sprintf("post-plot %x\n", tld.getptr()));

      saved_xcoord = xcoord;
     
      decor_bar--;

      if (decor_bar >= 0) { // skip on final loop iteration

        carry=0;
          
        if (selfmod_lda == 0) {
          xcoord += (tileslots[sm_tile_slot].vram_target >> 2) + 2;
          //trace3(sprintf("may not be unconditional %x\n", tld.getptr()));
          continue; // MAY NOT BE UNCONDITIONAL -- CHECK
        }
        // dr5_skip_selfmod_a
        // increment Y-coordinate
        w = tileslots[0].height + 1;
        carry = (w & 0x40) ? 1 : 0;
        ycoord += ((tileslots[0].height + 1 + carry) << 2);
        xcoord = saved_xcoord; // originally the accumulator
      
      }
      
      //trace3(sprintf("while decor_bar %x\n", tld.getptr()));
      
    } while (decor_bar >= 0);
    
    tiles_to_blit--; // blits_remaining
    //print "tiles remaining: ".tiles_to_blit."\n";
    
  } // end while()
  
  printf("\n");
  
  return CE_OK;
  
}


// 407f / nxt_roombyte_coord
// 4085 / coord_to_misc16_vp
static void room_byte_to_coordinates (u8_t rb, u8_t *x_out, u8_t *y_out) {
  *y_out = rb & 0xf0;        // y coordinate is on a pitch-16 grid
  *x_out = (rb & 0xf) * 10;  // x-coordinate is on a pitch-10 grid
}



static citerr_t room_rectangles (room_data_t *room, // Lrectangles
                                 s8_t num_rectangles,
                                 u8_t room_id,
                                 u8_t *water_X1_out,
                                 u8_t *water_X2_out,
                                 u8_t *water_level_out,
                                 u8_t star_port_destroyed,
                                 u8_t well_drained,
                                 vram_t *vram) {
                          
  u8_t rect_colour;
  u8_t rect_operation;
  s8_t rects_left;
  u8_t b;
  u8_t xc, yc;
  u8_t go_to_room_q_bit;
  
  // rectangles
  
  trace2 ("35c1 Lrectangles (room[M])");
  
  rect_colour = 0;

  printf("  + Room has %u rectangle%s:\n",
         (1 + num_rectangles), ((0==num_rectangles)?"":"s"));

  rect_operation = BLIT_STA;
  
  for (rects_left = num_rectangles;
       rects_left >= 0;
       rects_left--) { // P_rectangles -- this can even be jumped to from within room[Q]

    u8_t a, dummy, q, minY;
    u8_t num_mode2_blocks;
    s8_t num_mode2_stripes;
    
    a=0; // FIXME? need to move scope up a level?

    //trace2("t400_ptr_L (rem rects) = ".rects_left);
    
    //~ printf("35d7 --- P_rectangles_a: get another rectangle byte\n");

    // room[M+1]
    if (CE_OK != room_data_read(room, &b)) {
      return CE_TRUNC_ROOM_M_1;
    }
    
    // - if [M+1] high nybb is zero, it's a command byte? either a blitting operation looked up in
    //   T_SELFMOD10_OPCODES, or it's a special case for modifying the rectangle colour
    // - if [M+1] high nybb is nonzero, then low and high nybbles form X and Y coords in foo and bar:

    room_byte_to_coordinates(b, &xc, &yc);
    
    go_to_room_q_bit = 0;
    
    // zero y-coordinate seems to act as some sort of value
    // to do "special things"; a command byte:
    if (0 == yc) { // y-coord zero?
    
      // low nybble sets plot operation
      switch (b & 0xf) { // [M+1] low nybb is draw operation
        case 0: rect_operation = BLIT_STA; break;
        case 1: rect_operation = BLIT_ORA; break;
        case 2: rect_operation = BLIT_AND; break;
        case 3: rect_operation = BLIT_EOR; break;
      }
      
      // room[M+2]
      // command byte is followed by at least one more byte
      if (CE_OK != room_data_read(room, &b)) {
        return CE_TRUNC_ROOM_M_2;
      }
      a = b;
      
      // post-command byte 0x70 means "set up water stuff"
      if (a == 0x70) {
        // read two more bytes and use them to set up water parameters
        // so this whole sequence is like "0X 70 <water level> <water X-limits>" ?
        // room[M+3]
        if (CE_OK != room_data_read(room, &b)) {
          return CE_TRUNC_ROOM_M_3;
        }
        *water_level_out = b;
        // room[M+4]
        if (CE_OK != room_data_read(room, &b)) {
          return CE_TRUNC_ROOM_M_4;
        }
        *water_X1_out = 10 * (b & 0xf);
        *water_X2_out = 0xa + ((5 * (b & 0xf0)) / 8); // ???
      }
      
      // post-command byte 0x9 means ... ???
      if (a == 0x9) {
        a = room_id;
        // OK. this is a thing which modifies the Star Port
        // if it's the alien one, rather than the human planet one
        // if room_id >127, which corresponds to the human one,
        // clamp it to zero and do nothing
        if (a & 0x80) {
          a=0;
        } else {
          // room ID was in range 0-127
          // low room IDs mean "alien planet" ...
          // room 5, the alien one, doesn't have the steps on the left
          // skip room[P], room[P+1]
          if (CE_OK != room_data_read(room, &dummy)) { return CE_TRUNC_SKIP_P; }
          if (CE_OK != room_data_read(room, &dummy)) { return CE_TRUNC_SKIP_P; }
          // 3619
          if (RID_STAR_PORT_DISH_ALIEN == room_id) {
            xc = 0x14;
          } else { // probably applies to the alien star port dish room
            xc = 0x0;
          }
          yc = 0xd0;
          a = 0x13;
          // JUMP TO ROOM[Q] BIT
          go_to_room_q_bit = 1;
        }
      }
      
      // this block of code seems only to exist to special-case
      // rectangle colours
      if ( ! go_to_room_q_bit ) {
      
        // colour 0xcb has collision=binary '11' (impermeable);
        // left pixel is yellow, right pixel is red;
        // this is the Star Port vertical striping ...
        if ((a == 0xcb) && star_port_destroyed) { // && (r_inout.puzzle_flags_10 != 0)) {
          // colour is changed to f2
          // f2 has binary '11' collision again, but now
          // left pixel is purple, right pixel is blue ...
          a = 0xf2;
        }

        // a == 0xf0 means colour is blue water
        if ((a == 0xf0) && (room_id == RID_WELL) && well_drained) { //p_inout.t_puzzle_flags[2]) {
          // if the well is drained, water does not have impermeable
          // collision, and is no longer blue!
          a = 0;
        }
        
        rect_colour = a; // rectangle colour
        
        // hack: t400_ptr_L is only decremented at the end of the loop,
        // so if we do this early continue, we have to increment it
        // to simulate it not being decremented ...
        rects_left++;
        continue; // jump back to P_rectangles
        
      }
      
    } // endif (ycoord was zero)
   
    // if we got here without executing if (bar was nonzero) { } block,
    // then we have to read another byte; if not, then we have the byte already
    if ( ! go_to_room_q_bit ) {
      // room[Q]
      if (CE_OK != room_data_read (room, &q)) {
        return CE_TRUNC_ROOM_Q;
      }
    } else {
      q = a;
    }
   
    minY = ((VRAM_GFX_WINDOW_START >> 8) & 0xff) - 1;
    
    num_mode2_blocks = q & 0xf;
    
    if (yc < minY) { // stop rogue Y-coordinates from underflowing VRAM
      yc = minY; // clamp
      num_mode2_stripes = (q >> 3) & 0xfe;
    } else {
      num_mode2_stripes = (q >> 3) | 1;
    }
    
    //trace ("3662 before P_rect_blit_outer\n");
    
    //~ printf("_outer: tileptr_H=%x\n", p_inout.tileptr_H);
    
    // room[Q] is the "hollowing-out" code
    // blits a rectangle, usually black but that's dictated by rect_colour, of any width and height in any position
//      print "      (".xc.", ".yc."), ".((1+num_mode2_blocks)*10)." x ".((1+num_mode2_stripes)*8)."\n";
    printf("      (%3u, %3u), [%3u x %3u], colour 0x%x\n",
           xc, yc, ((1+num_mode2_blocks)*10), ((1+num_mode2_stripes)*8), rect_colour);
    
    for ( ;
         num_mode2_stripes >= 0;
         num_mode2_stripes--, yc += 8) { // _outer:
         
      u16_t vram_ptr;
      s8_t x, y;
      citerr_t e;
      
      vram_ptr = coords_to_vram_ptr(xc, yc); // recompute coords
      
//printf("vram_ptr = %x\n", vram_ptr);
      
      for (x = num_mode2_blocks;
           x >= 0;
           x--, vram_ptr += 0x28) { // _inner:
           
        for (y = 0x27; y >= 0; y--) { // _hard: 40 bytes, 80 pixels, 10x8 block
        
          if (DEBUG_ENABLE_RECTANGLES) {
          
            e = vram_op (vram,
                         rect_operation, // SM_rect_operation
                         rect_colour,    // SM_rect_colour
                         vram_ptr + y);
                        
            if (CE_OK != e) { return e; }
            
          }
          
        }
        
      }
      
    }
    
    //~ trace2("368c t400_ptr_L = 0x".sprintf("%x", 0xff & rects_left));
    
  } // otherwise jump back to P_rectangles_a
  
  //trace2("3693 BEGIN TRIANGLES");
  //~ if (p_inout.decor_bar & 7) {
    //~ p_inout.decor_bar &= 7;
  //~ } else {
    //~ skip_trgl_stuff = 1; // returned
  //~ }
  
  return CE_OK;
  
}

static citerr_t room_triangles (room_data_t *room,
                                u8_t star_port_destroyed,
                                u8_t *num_triangles_inout, // pass in from (decor_bar & 0x7)
                                triangle_t trgls_inout[MAX_TRIANGLES]) {
  
  // handles triangles
  // starts at 0x3693
  
  triangle_t wt; // working_triangle
  s8_t curr_trgl_ix;
  //s8_t i;
  s8_t num_triangles;
  
  num_triangles = *num_triangles_inout; // duplicate this
  
  *num_triangles_inout = 0;
  
  //trace ("3693 room_r triangles");
  
//printf("room_triangles(%d)\n", num_triangles);
  
  if (0 == num_triangles) {
    return CE_OK; // Lskipped_triangle_stuff
  }
  
  /*
  memset(trgls_inout, 0, sizeof(triangle_t) * MAX_TRIANGLES);
  
  for (i=0; i < num_triangles; i++) {
    trgls_inout[i].fg_plot_mode = -1;
    trgls_inout[i].fg_colour = -1;
  }
  */
  

  
  for (curr_trgl_ix = 0;
       curr_trgl_ix < num_triangles;
       curr_trgl_ix++) { // P_triangles

//printf("[1]\n"); 
  
    u8_t b, c;
    u8_t sx, sy;
    u8_t room_s_plus_one;
    u8_t room_r_high_nybble;
    citerr_t e;
    //triangle_t *t;
        
    // this is the top section that sets up plot modes and colour changes
    // runs until the room byte read [R] has a nonzero high nybble
    
    // -- FIXME: this is some nasty ASM control flow; really all that happens
    //    here is:
    // - read a room byte R
    // - if R has high nybble set:
    //   - read a second byte, where [R] low and [R+1] give plot mode and colour
    //   - read a third byte, and this is just the new R which replaces the original one
    // - carry on with whichever R is relevant; there need be no top loop here
    
    triangle_init (&wt);
    
    //t = trgls_inout + curr_trgl_ix;
    
    do {
    
//printf("[2]\n");
      
      // room[R]
      //b = room.read(CE_TRUNC_ROOM_R);
      if (CE_OK != room_data_read (room, &b)) {
        return CE_TRUNC_ROOM_R;
      }
      
      // FIXME: don't use this stupid variable,
      // use a local instead
      room_r_high_nybble = b & 0xf0; // high nybble of room[R], "RH"
      
      if ((b & 0xf0) == 0) {
      
        // this is "foreground plot mode"
        //~ t->fg_plot_mode = b & 0xf;
        wt.fg_plot_mode = b & 0xf;
        
        // room[R+1]
        if (CE_OK != room_data_read (room, &c)) {
          return CE_TRUNC_ROOM_R_PLUS;
        }
        
        wt.fg_colour = c;
        
      }
      
    } while ((b & 0xf0) == 0); // FIXME: silly; see above
    
    // now we have either the original room[R], or a replacement one after
    // changing plot mode and colour
    
    // 36bc
    //~ t->v1.x = (10 * (b & 0xf));         // new: aka "RL"
    wt.v1.x = (10 * (b & 0xf));         // new: aka "RL"

    //print "36bc v1.x=".sprintf("%x\n", working_triangle.v1.x);
    
    // room[S] (which is either [R+1] or [R+3])
    if (CE_OK != room_data_read (room, &c)) {
      return CE_TRUNC_ROOM_S;
    }
    
    // new
    //~ t->v2.x = 10 * (c & 0xf);
    wt.v2.x = 10 * (c & 0xf);
    //print "36c2 v2.x=".sprintf("%x\n", working_triangle.v2.x);
    
    //~ t->v2.y = 0xff & ~(8 + ((c & 0xf0) - 1));
    wt.v2.y = 0xff & ~(8 + ((c & 0xf0) - 1));
    
    //print "36cb v2.y=".sprintf("%x\n", working_triangle.v2.y);
    
    // room[S+1]
    if (CE_OK != room_data_read (room, &room_s_plus_one)) {
      return CE_TRUNC_ROOM_S_PLUS;
    }

    //print "36dc A=".sprintf("%x\n", room_s_plus_one);
   
    // &CC is a sentinel value which stops triangle parsing after this point,
    // if t_puzzle_flags_2[0] is set:
    if ((room_s_plus_one != 0xcc) || (star_port_destroyed == 1)) { // 36d2
    
      room_byte_to_coordinates (room_s_plus_one, &sx, &sy);
      
      if (sx < (wt.v1.x)) {
        // some sort of correction?
        //~ t->v1.x--;
        wt.v1.x--;
      }
      //print "36e5 v1.x = ".sprintf("%x\n", working_triangle.v1.x);

      if (sx < (wt.v2.x)) {
        // some sort of correction?
        wt.v2.x += 9;
      }

      //~ print "36f2 v2.x = ".sprintf("%x\n", working_triangle.v2.x);
      
      // alt_tileptr is the saved "RH" from before, the room[R] high nybble
      // we shifted it into the high byte, so like Nxxx
      
      if (sy != room_r_high_nybble) {
        // decrement alt_tileptr high byte... WHY?
        room_r_high_nybble--;
      }
      
      wt.v3_tmp.x_high = 0xff & (sx >> 5);
      wt.v3_tmp.x_low  = 0xff & (sx << 3);
      
      wt.v3_tmp.y_high = 0xff & ((0xff & ~sy) >> 6);
      wt.v3_tmp.y_low  = 0xff & ((0xff & ~sy) << 2);
      
      //371d:                     adc #08
      //371f:                     eor #ff
      //3721:                     sta _cur_gfx_cursor_Y_L

      room_r_high_nybble += 8; // FIXME: plus carry from stuff above ...
      wt.v1.y = 0xff & ~(room_r_high_nybble);
      
      // fix up point v3 to use internal coordinates
      // and compensate for the nonzero graphical origin (0, -32)
      // also fix y-coords (y = 175 - y)
      e = triangle_fix_coords(&wt);
      if (CE_OK != e) { return e; }
      
      // copy trgl to output
      trgls_inout[*num_triangles_inout] = wt;
      (*num_triangles_inout)++;
      
      // VDU 25 would go here ...

    }
    
  } // next triangle
  
  return CE_OK;
  
}


static citerr_t room_item_pad (room_data_t *room,
                               s16_t item_tile_id,
                               u8_t *item_pad_X_out,
                               u8_t *item_pad_Y_out,
                               u8_t plot_mode,
                               tile_data_t *tiledata,
                               vram_t *vram,
                               u8_t *pixel_base_pairs,
                               u8_t *animation_speeds) {
                               
  s8_t tile_width_maybe_dummy;
  u8_t t_500[T_500_LEN];
  u8_t t_xyzzy[4];
  u8_t tile_slot_ix;
  tile_slot_t tileslots_local[NUM_TILESLOTS];
  citerr_t e;
  u8_t pad;
  u8_t carry_dummy;
  u8_t hi, lo_10;

  // handles item pad position
  //trace ("3733 room_item_pad");
  
//printf("room_item_pad\n");
  
  tile_width_maybe_dummy = 0;
  
  memset(t_500, 0, T_500_LEN);
  memset(t_xyzzy, 0, 4);

  // get the item pad gfx
  // we reinitialise this, which might not be what the 6502 does
  //t_500 = array();
  //t_xyzzy = array();
  tile_slot_ix = 0; //xff;
  
  tile_data_wipe400(tiledata);
  
  e = unpack_tile_gfx (tiledata,
                       TID_ITEM_PAD,
                       t_500,
                       t_xyzzy,
                       &tile_slot_ix,
                       0xff, 0xff,
                       0xff, 0xff, // tile_500buf_pos_1_initial, .._2_initial
                       &tile_width_maybe_dummy,
                       tileslots_local,
                       1, // initialise
                       pixel_base_pairs,
                       animation_speeds,
                       &carry_dummy);
  if (CE_OK != e) { return e; }

  // grab coordinates from bytestream
  if (CE_OK != room_data_read (room, &pad)) {
    return CE_TRUNC_ITEM_PAD_POS;
  }
  
  hi = 0xf0 & pad;
  lo_10 = 10 * (0xf & pad);
  
  *item_pad_Y_out = hi - 16;
  // some hack for some room? check which
  *item_pad_X_out = lo_10 ? lo_10 : 0x4a; // X = 0x4a default
  
  // draw pad
  e = tile_plot_specify_tile_slot(tiledata,
                                  tileslots_local + 0,
                                  t_500,
                                  plot_mode,
                                  *item_pad_X_out,
                                  *item_pad_Y_out + 16,
                                  0, // force branch B
                                  0, // entry point H
                                  0, // entry point H areg
                                  vram);

  tile_width_maybe_dummy = 0;

  if (item_tile_id >= 0) {
  
    // draw item on pad
    
    // unpack it
    unpack_tile_gfx(tiledata,
                    0xff & item_tile_id,
                    t_500,
                    t_xyzzy,
                    &tile_slot_ix,
                    0xff, 0xff,
                    0xff, 0xff, // tile_500buf_pos_1_initial, .._2_initial
                    &tile_width_maybe_dummy,
                    tileslots_local,
                    1, // initialise => tile slot 0
                    pixel_base_pairs,
                    animation_speeds,
                    &carry_dummy);

    // plot it
    //printf("Plotting item on pad at (%x, %x)\n", p_inout.item_pad_X, p_inout.item_pad_Y);
    // variant that (a) forces tile slot 0 and (b) diddles with the coordinates:
    e = tile_plot_fix_coords_tile_slot_0 (tiledata,
                                          tileslots_local + 0,
                                          t_500,
                                          plot_mode,
                                          *item_pad_X_out,
                                          *item_pad_Y_out,
                                          vram);
    if (CE_OK != e) { return e; }

  }
  
  return CE_OK;
  
}


// variant; something to do with pre-adjusting coordinates
// based on the size of the tile in slot 0, and the target position?
static citerr_t tile_plot_fix_coords_tile_slot_0 (tile_data_t *tiledata,
                                                  tile_slot_t *ts0,
                                                  u8_t t_500[T_500_LEN],
                                                  u8_t plot_mode,
                                                  u8_t x,
                                                  u8_t y,
                                                  vram_t *vram) {
                                           
  u8_t a, carry;
  u8_t vram_targ_low;
  citerr_t e;

  // 2577

  // implicitly tile slot 0
  a = (3 - ts0->height); // 257a
  carry = a & 0x40;
  a <<= 2;
  y += a + carry; // 257f
  vram_targ_low = (ts0->vram_target & 0xff);
  
  // hm. what's so special about vram_ptr low and 0x13?
  if (vram_targ_low == 0x13) { // 2586
    x += 2;
  } else if (vram_targ_low < 0x13) {
    x += 4;
  }
  
  e = tile_plot_a_slot_0 (tiledata,
                          ts0,
                          t_500,
                          plot_mode,
                          x,
                          y,
                          vram);
                          
  return e;
  
}
