#include "vram.h"

#include <string.h>
#include <stdio.h>

static citerr_t mode2_to_lin (u16_t addr, s32_t limit, s32_t *lin_out);
static void map_collision_colour (u8_t col_colour, u8_t *r_out, u8_t *g_out, u8_t *b_out);

void vram_init (vram_t *vram, u8_t height) {
  // TODO: no-background mode
  vram->linlen = height * FRAMEBUF_WIDTH * 4;
  memset(vram->lincol, 0, vram->linlen);
  memset(vram->linvis, 0, vram->linlen);
}

citerr_t vram_write (vram_t *vram, u16_t pos, u8_t byte) { // MODE 2 representation

  s32_t linpos;
  u8_t gfx_left, gfx_right, col_left, col_right;
  u8_t r,g,b;
  u8_t col_colour;
  citerr_t e;
  
  if (pos < VRAM_GFX_WINDOW_START) {
    // ignore minor underflow writes (ropes do this)
    return CE_OK;
  }

  if (pos < VRAM_START_HARD_LIMIT) {
    printf ("ERROR: vram write underflow (pos 0x%x, start 0x%x)\n",
            pos, VRAM_GFX_WINDOW_START);
    return CE_VRAM_WRITE_BOUNDS;
  }

  pos -= VRAM_GFX_WINDOW_START; // adjust for zero-indexed
  
  if (pos >= VRAM_LENGTH) { // write to ROM; ignore
    return CE_OK;
  }

  // get linear position (lefthand of two pixels)
  e = mode2_to_lin (pos, vram->linlen - 8, &linpos);
  if (CE_OK != e) { return e; }
  
  // graphics; 0-7, BGR
  gfx_left  = ((byte >> 3) & 4) | ((byte >> 2) & 2) | ((byte >> 1) & 1);
  gfx_right = ((byte >> 2) & 4) | ((byte >> 1) & 2) | (byte & 1);
  
  // collision; 0-1
  col_left  = ((byte >> 7) & 1);
  r = (gfx_left & 1) ? 0xff : 0;
  g = (gfx_left & 2) ? 0xff : 0;
  b = (gfx_left & 4) ? 0xff : 0;
  //a = 0xff;
  
  vram->linvis[linpos]   = r;
  vram->linvis[linpos+1] = g;
  vram->linvis[linpos+2] = b;
  vram->linvis[linpos+3] = 0xff;
  
  col_right = ((byte >> 6) & 1);
  r = (gfx_right & 1) ? 0xff : 0;
  g = (gfx_right & 2) ? 0xff : 0;
  b = (gfx_right & 4) ? 0xff : 0;
  //a = 0xff;
  
  vram->linvis[linpos+4] = r;
  vram->linvis[linpos+5] = g;
  vram->linvis[linpos+6] = b;
  vram->linvis[linpos+7] = 0xff;
  
  // collision will be done slightly differently
  // as it effectively has half resolution
  // get a colour based on a combination of both
  // left and right pixels -- essentially a 2-bit colour
  col_colour = ((col_left << 1) & 2) | col_right;
  
  map_collision_colour(col_colour, &r, &g, &b);
  
  vram->lincol[linpos]   = r;
  vram->lincol[linpos+1] = g;
  vram->lincol[linpos+2] = b;
  vram->lincol[linpos+3] = 0xff;
  vram->lincol[linpos+4] = r;
  vram->lincol[linpos+5] = g;
  vram->lincol[linpos+6] = b;
  vram->lincol[linpos+7] = 0xff;
  
  return CE_OK;
  
}

static void map_collision_colour (u8_t col_colour, u8_t *r_out, u8_t *g_out, u8_t *b_out) {
  switch (col_colour) {
    case 1:  *r_out = 255; *g_out = *b_out = 0; break;
    case 2:  *r_out = *b_out = 0; *g_out = 255; break;
    case 3:  *r_out = *g_out = 255; *b_out = 0; break; // does this ever happen?
    default: *r_out = *g_out = *b_out = 0;      break;
  }
}

citerr_t vram_read (vram_t *vram, u16_t pos, u8_t *out) {

  citerr_t e;
  s32_t linpos;
  u8_t r_left, g_left, b_left;
  u8_t r_right, g_right, b_right;
  u8_t c_left, c_right;
  u8_t mode2_left, mode2_right;
  
  // ropes seem to read-underflow VRAM_GFX_WINDOW_START, so we'll
  // have to set a hard limit that lies below VRAM_GFX_WINDOW_START
  if (pos < VRAM_START_HARD_LIMIT) {
    printf ("ERROR: vram read underflow (pos 0x%x, hard low limit 0x%x)\n", pos, VRAM_START_HARD_LIMIT);
    return CE_VRAM_READ_BOUNDS;
  }

  // .. and just ignore minor underflows
  if (pos < VRAM_GFX_WINDOW_START) {
    //printf ("ERROR: vram read underflow (pos 0x%x, start 0x%x)\n", pos, VRAM_GFX_WINDOW_START);
    //return CE_VRAM_READ_BOUNDS;
    return CE_OK;
  }

  pos -= VRAM_GFX_WINDOW_START; // adjust for zero-indexed
  
  if (pos >= VRAM_LENGTH) {
    return CE_OK;
    //printf("ERROR: read of ROM area (0x%x bytes into VRAM)\n", pos);
    //return CE_VRAM_READ_BOUNDS;
    // return 0;
  }
  
  linpos = 0;

  // get linear position (lefthand of two pixels)
  e = mode2_to_lin(pos, vram->linlen - 8, &linpos);
  
  if (linpos >= vram->linlen) {
    *out = 0;
    // return -1; // ROM; ignore
    return CE_OK;
  }
  
  r_left  = vram->linvis[linpos];
  g_left  = vram->linvis[linpos+1];
  b_left  = vram->linvis[linpos+2];
  // don't care about alpha
  r_right = vram->linvis[linpos+4];
  g_right = vram->linvis[linpos+5];
  b_right = vram->linvis[linpos+6];
  // don't care about alpha
  //~ c_left  = ord(this.lincol[linpos]);
  //~ c_right = ord(this.lincol[linpos+4]);
  c_left  = vram->lincol[linpos+1]; // "green"
  c_right = vram->lincol[linpos];   // "red"
  
  // components are all 0xff or 0 so we can just mask without shifting and
  // we're guaranteed to get ones in the correct places
  mode2_left  = (c_left  & 0x80) | (b_left  & 0x20) | (g_left  & 0x8) | (r_left  & 0x2);
  mode2_right = (c_right & 0x40) | (b_right & 0x10) | (g_right & 0x4) | (r_right & 0x1);
  
  *out = (mode2_left | mode2_right);
  
  return e;
  
}


citerr_t vram_linplot (vram_t *vram,
                       s16_t x,
                       s16_t y,
                       u8_t r,
                       u8_t g,
                       u8_t b,
                       u8_t c) { // 0-3
  s32_t p;
  p = (x + (FRAMEBUF_WIDTH * y)) * 4;
//printf("linplot: p = %x\n", p);
  if (p >= vram->linlen) {
    printf("ERROR: vram_linplot(): overflow (%d)\n", p - vram->linlen);
    return CE_VRAM_BOUNDS;
    //return CE_OK;
  }
  if (p < 0) {
    //printf("WARNING: vram_linplot(): underflow (%d)\n", p);
    //return CE_VRAM_BOUNDS;
    return CE_OK;
  }
  vram->linvis[ p ] = r;
  vram->linvis[p+1] = g;
  vram->linvis[p+2] = b;
  vram->linvis[p+3] = 0xff;
  map_collision_colour(c, &r, &g, &b);
  vram->lincol[ p ] = r;
  vram->lincol[p+1] = g;
  vram->lincol[p+2] = b;
  vram->lincol[p+3] = 0xff;
  return CE_OK;
}


// blit operation
citerr_t vram_op (vram_t *vram, u8_t blit_mode, u8_t value, u16_t addr) {
  u8_t old_value, new_value;
  citerr_t e;
  e = vram_read (vram, addr, &old_value);
  if (CE_OK != e) { return e; }
  switch (blit_mode) {
    case BLIT_STA: new_value = value;             break;
    case BLIT_EOR: new_value = old_value ^ value; break;
    case BLIT_AND: new_value = old_value & value; break;
    case BLIT_ORA: new_value = old_value | value; break;
    default:
      printf("ERROR: vram_op(): bad blit mode (%u)\n", blit_mode);
      return CE_BAD_BLIT_MODE;
  }
  return vram_write (vram, addr, new_value);
}


citerr_t vram_copy2px (vram_t *vram, u16_t dstaddr, u16_t srcaddr) {
  citerr_t e;
  u8_t px;
  e = vram_read(vram, srcaddr, &px);
  if (CE_OK != e) { return e; }
//printf("%x\n", dstaddr); exit(0);
  return vram_write (vram, dstaddr, px);
}


static citerr_t mode2_to_lin (u16_t addr, s32_t limit, s32_t *lin_out) {

  s16_t sn, sp, bn, bp;
  s16_t x, y;
  s32_t pxix;
  s32_t result;
  u16_t a;

  // (mode2addr/640) gives stripe number, SN (Y)
  // (         %640) gives stripe position, SP (X&Y)
  // (SP / 8) gives band number, BN (X)
  // (SP % 8) gives band position, BP (Y)
  
  // linear position is given by
  // y = (SN * 8) + (BP)
  // x = (BN * 2)
  
  a = addr;
  
  sn = a / 640;
  sp = a % 640;
  bn = sp / 8;
  bp = sp % 8;

  y = (sn * 8) + bp;
  x = (bn * 2);
  
  pxix = x + (y * FRAMEBUF_WIDTH);
  
  result = pxix * 4; // 4 bytes pp
  
  if (result > limit) {
    printf("ERROR: mode2_to_lin(): linear position (%x) exceeds limit (%x)\n",
           result, limit);
    return CE_VRAM_BOUNDS;
  }
  
  *lin_out = result;
  
  return CE_OK;

}
