/*
 *  Quadbike 2
 *  Copyright (C) 2023 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "fir.h"

#include <string.h>
#include <stdio.h>

#ifdef QB_SANITY

u8_t qb_validate_sample_rate_ix (u8_t rate_ix) {
  return    (QB_FIR_RATE_22K == rate_ix)
         || (QB_FIR_RATE_44K == rate_ix)
         || (QB_FIR_RATE_48K == rate_ix);
}
#endif // QB_SANITY

qb_err_t qb_fir_init (qb_fir_t *f, u8_t type, u8_t rate) {

  u32_t i;
  
  memset(f, 0, sizeof(qb_fir_t));

#ifdef QB_SANITY
  if ( ! qb_validate_sample_rate_ix (rate) ) {
    fprintf(stderr, "B: %s: bad sample rate ix %u\n", QB_FUNC_M, rate);
    return QB_E_BUG;
  }
#endif
  
  // bandpass 400->3200 is for the -f option
  if (QB_FIR_TYPE_BP_0K4_3K2 == type) {
    if (QB_FIR_RATE_22K == rate) {
      f->num_taps = QB_FIR_TAPS_BP_0K4_3K2_22K;
      f->taps = qb_fir_taps_bp_0k4_3k2_22k;
      f->delay = QB_FIR_SMPSDELAY_BP_0K4_3K2_22K;
    } else if (QB_FIR_RATE_44K == rate) {
      f->num_taps = QB_FIR_TAPS_BP_0K4_3K2_44K;
      f->taps = qb_fir_taps_bp_0k4_3k2_44k;
      f->delay = QB_FIR_SMPSDELAY_BP_0K4_3K2_44K;
    } else if (QB_FIR_RATE_48K == rate) {
      f->num_taps = QB_FIR_TAPS_BP_0K4_3K2_48K;
      f->taps = qb_fir_taps_bp_0k4_3k2_48k;
      f->delay = QB_FIR_SMPSDELAY_BP_0K4_3K2_48K;
    }
  // bandpass-2400 is applied to the extracted carrier
  // before it is fed to the PLL
  } else if (QB_FIR_TYPE_BP_2K4 == type) {
    if (QB_FIR_RATE_22K == rate) {
      f->num_taps = QB_FIR_TAPS_BP_2K4_22K;
      f->taps = qb_fir_taps_bp_2k4_22k;
      f->delay = QB_FIR_SMPSDELAY_BP_2K4_22K;
    } else if (QB_FIR_RATE_44K == rate) {
      f->num_taps = QB_FIR_TAPS_BP_2K4_44K;
      f->taps = qb_fir_taps_bp_2k4_44k;
      f->delay = QB_FIR_SMPSDELAY_BP_2K4_44K;
    } else if (QB_FIR_RATE_48K == rate) {
      f->num_taps = QB_FIR_TAPS_BP_2K4_48K;
      f->taps = qb_fir_taps_bp_2k4_48k;
      f->delay = QB_FIR_SMPSDELAY_BP_2K4_48K;
    }
  }
  
  if (0 == f->num_taps) {
    fprintf (stderr, "B: %s: bad FIR type %u for rate %u\n", QB_FUNC_M, type, rate);
    return QB_E_BUG;
  }
  
  for (i = 0; i < f->num_taps; i++) {
    f->history[i] = 0.0;
  }
  
  f->last_index = 0;
  
  //~ printf("Initialised FIR filter; type %u, taps %u, delay %u.\n",
         //~ type, f->num_taps, f->delay);
  
  return QB_E_OK;
  
}


void qb_fir_in (qb_fir_t *f, float input) {
  f->history[f->last_index] = input; // float -> double
  f->last_index++;
  if (f->num_taps == f->last_index) {
    f->last_index = 0;
  }
}


float qb_fir_out (qb_fir_t *f) { // double internally, float externally

  double acc;
  u32_t index;
  u32_t i;
  
  acc = 0.0;
  index = f->last_index;
  
  for (i = 0; i < f->num_taps; i++) {
    index = (index != 0) ? (index - 1) : (f->num_taps - 1);
    acc += f->history[index] * f->taps[i];
  }
  
  return (float) acc;
  
}

#include "util.h"

u8_t qb_sample_rate_ix_from_rate (s32_t rate) {
  qb_err_t e;
  e = qb_test_rate_legal (rate);
  if (QB_E_OK != e) {
    return QB_FIR_RATE_UNK;
  }
  if (22050 == rate) {
    return QB_FIR_RATE_22K;
  } else if (44100 == rate) {
    return QB_FIR_RATE_44K;
  } else if (48000 == rate) {
    return QB_FIR_RATE_48K;
  }
  return QB_FIR_RATE_UNK;
}




