/*
 *  Quadbike 2
 *  Copyright (C) 2023 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "polarity.h"
#include "span2.h"
#include "qb_err.h"
#include "atom.h"

#include <math.h>

static qb_err_t does_span_invert_polarity (qb_span_t *span,
                                           u8_t *yes_out,
                                           s64_t *num_inverting_runs_out,
                                           s64_t *num_silent_cycles_out);


static qb_err_t does_span_invert_polarity (qb_span_t *span,
                                           u8_t *yes_out,
                                           s64_t *num_inverting_runs_out,
                                           s64_t *num_silent_cycles_out) {

  s32_t an;
  u8_t consecutive_0_cycles;

  *yes_out = 0;
  *num_inverting_runs_out = 0;
  *num_silent_cycles_out = 0;
  
  consecutive_0_cycles = 0;
  
  for (an=0; an < span->num_atoms; an++) {
  
    qb_atom_t *cyc;
    
    cyc = span->atoms + an;
    
    // (a silent cycle is just the same thing as a zero cycle really,
    // as far as the CSW is concerned)
    if (('0' == cyc->value) || ('S' == cyc->value)) {
      consecutive_0_cycles++;
      if ('S' == cyc->value) {
        (*num_silent_cycles_out)++;
      }
    } else { //if (('1' == cyc->value) || ('L' == cyc->value)) {
      // end of zero-run
      if (consecutive_0_cycles & 1) {
        (*num_inverting_runs_out)++;
      }
      consecutive_0_cycles = 0;
    }
    
  } // next cycle
  
  // possibly a final one ...
  if (consecutive_0_cycles & 1) {
    (*num_inverting_runs_out)++;
  }
  
  if (*num_inverting_runs_out & 1) {
    *yes_out = 1;
  }
  
  return QB_E_OK;
  
}



// called immediately before CSW write ...
qb_err_t qb_spans_scan_polarities (qb_span_t *spans,
                                  s32_t num_spans,
                                  u8_t *at_least_one_inversion_required_out,
                                  u8_t setup_phase,
                                  u8_t verbose) {
                               
  s32_t sn;
  qb_err_t e;
  u8_t header_printed;
  u8_t first_print;
  
  e = QB_E_OK;
  if (NULL != at_least_one_inversion_required_out) {
    *at_least_one_inversion_required_out = 0;
  }
  
  first_print = 1;
  
  // search for odd zero-runs (which are what cause polarity reversals);
  // mark spans that have an odd number of such runs (which reverse polarity);
  
  for (sn=1; sn < num_spans; sn++) {
  
    qb_span_t *span;
    s64_t num_inverting_runs;
    s64_t num_silent_cycs;
    char c;
    const char *text;
    
    num_inverting_runs = 0;
    span = spans + sn;
    
    e = does_span_invert_polarity (span,
                                   &(span->reverses_output_polarity),
                                   &num_inverting_runs,
                                   &num_silent_cycs);
    if (QB_E_OK != e) { return e; }
    
//printf("span %d: inverts polarity = %u\n", sn, span->reverses_output_polarity);
    
    if ((NULL != at_least_one_inversion_required_out) && span->reverses_output_polarity) {
      *at_least_one_inversion_required_out = 1;
    }
    
    if (qb_is_squawk (span, (span-1))) {
      text = "squawk";
    } else {
      text = qb_span_get_type_text(span);
    }

    if (span->reverses_output_polarity) {
      if (setup_phase) {
        if (verbose) {
          fprintf(QB_ERR,
                  "  W: [%lld] %s span #%d flips polarity; %lld odd run%s, %lld silent cyc%s\n",
                  qb_get_span_start_accurate_or_rough(span),
                  text,
                  sn,
                  num_inverting_runs,
                  (num_inverting_runs != 1) ? "s" : "",
                  num_silent_cycs,
                  (num_silent_cycs != 1) ? "s" : "");
        }
      } else { // DEPRECATED: this is never actually printed (we're always called with setup_phase = 1)
        if ( first_print ) {
          printf("  Invert-o-spans:\n    ");
        }
        if (QB_SPAN_TYPE_DATA == span->type) {
          c = 'D';
        } else if (QB_SPAN_TYPE_LEADER == span->type) {
          c = 'L';
        } else {
          c = 'S';
        }
        printf("%s%c%u", (first_print ? "" : ", "), c, sn);
        first_print = 0;
      }
    }
    
  } // next span
  
  if ( ! setup_phase) {
    if ( ! first_print ) {
      printf("\n");
    }
    return QB_E_OK;
  }
  
  for (sn=0; sn < num_spans; sn++) {
  
    qb_span_t *span;
    
    span = spans + sn;
    
    // hmm
    // - if a data span reverses polarity, we need to correct that *after*
    //   the data span, so it doesn't mess up the rest of the file.
    // - leader and silent spans should never reverse polarity
    
    if ((span->reverses_output_polarity) && (sn < (num_spans - 1))) {
      // correct data spans at start of *next* span
      // sanity
      if (QB_SPAN_TYPE_DATA == (span+1)->type) {
        fprintf(QB_ERR, "B: %s: Two adjacent data spans?!\n", QB_FUNC_M);
        return QB_E_BUG;
      }
      // flip to-invert flag on subsequent span
      (span+1)->need_to_invert = ! (span+1)->need_to_invert;
    }

  } // next span
  
  // print strategy ...
  header_printed=0;
  for (sn=0; sn < num_spans; sn++) {
    if (spans[sn].need_to_invert) {
      if ( ! header_printed ) {
        header_printed = 1;
        //printf("Output polarity correction strategy:\n");
      }
      if (QB_SPAN_TYPE_DATA != spans[sn].type) {
        if (verbose) {
          printf("    - [%lld] squeeze extra cycle into %s span #%d\n",
               qb_get_span_start_accurate_or_rough(spans + sn),
               qb_span_get_type_text(spans + sn),
               sn);
        }
      } else {
        // sanity check
        fprintf(QB_ERR, "B: %s: somehow trying to reverse polarity on a data span? should be impossible\n", QB_FUNC_M);
        return QB_E_BUG;
      }
    }
  }
  
  //printf("QUALITY: %lld singleton zeros.\n", singleton_zeros);
  
  return e;
  
}

qb_err_t qb_correct_polarities (qb_span_t *spans,
                                s32_t num_spans,
                                s32_t rate,
                                s64_t total_input_len,
                                u8_t verbose) {

  s32_t sn;
  qb_err_t e;
  
  e = QB_E_OK;
  
  for (sn=0; sn < num_spans; sn++) {
  
    qb_span_t *span;
    
    span = spans + sn;
    
    if ( ! span->need_to_invert ) {
      continue;
    }
    
    // use floating point copy of sample_num field
    // copy on
    qb_span_deal_w_float_smpnum_field (span, 1);
    
    span->need_to_invert = 0; // reset flag
    
    do {
      
      if (QB_SPAN_TYPE_DATA == span->type) {
        fprintf(QB_ERR,
                "B: %s: [%lld] need_to_invert is set on data span #%d!\n",
                QB_FUNC_M, qb_get_span_start_accurate_or_rough(span), sn);
        e = QB_E_BUG;
        break;
      } else if (QB_SPAN_TYPE_LEADER == span->type) {
        e = qb_span_insert_atom (span,
                               (sn < (num_spans-1))?(span+1):NULL,
                               total_input_len,
                               0, // 0 = insert at first cycle of leader span (this could be anywhere)
                               (s64_t) (round((double) rate) / (QB_FREQ_2 * span->speed)), // use nominal len
                               '0', // value is '0', as this will force a single pulse
                               verbose);
        if (QB_E_OK != e) { break; }
      } else {
        // silent span
        // silent span already contains 2 cycles, so we need to add a third
        // which will just be 1/3 of the length of the span
        if (span->len < 3) {
          fprintf(QB_ERR,
                  "E: [%lld] silent span #%d has len < 3 smps; no room to insert inverting pulse!\n",
                  qb_get_span_start_accurate_or_rough(span), sn);
          return QB_E_SILENT_SPAN_TOO_SHORT_FOR_INVERT;
        }
        e = qb_span_insert_atom (span,
                               (sn < (num_spans-1))?(span+1):NULL,
                               total_input_len,
                               0, // 0 = insert at first cycle of silent span (this could be anywhere)
                               (span->len / 3),
                               '0', // value is '0', as this will force a single pulse
                               verbose);
        if (QB_E_OK != e) { break; }
      }
      
    } while (0);
    
    // copy off
    qb_span_deal_w_float_smpnum_field (span, 0);
    
    if (QB_E_OK != e) { break; }
    
  } // next span
  
  return e;
  
}
