/*
 *  Quadbike 2
 *  Copyright (C) 2025 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "pll.h"
#include "util.h"
#include "qbio.h" // for debug file writing

#include <string.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>



static double pll_filter_process (qb_pll_filter_t *f, double in);

static void pll_filter_init (qb_pll_filter_t *filter,
                             double frequency,
                             u32_t sample_rate,
                             double q_factor);

void qb_pll_init (qb_pll_t *pll,
                  s32_t sample_rate,
                  double loop_cutoff_hz,
                  double lock_cutoff_hz,
                  double loop_gain,
                  double pll_cf, // centre frequency (Hz)
                  double lock_thresh) { // display progress

  double one_over_root_two;
  //qb_err_t e;
  
  memset(pll, 0, sizeof(qb_pll_t));
  
  one_over_root_two   = 1.0 / sqrt(2.0);
  
  pll->sample_rate    = (float) sample_rate;  
  pll->loop_gain      = loop_gain;
  pll->pll_cf         = pll_cf;
  pll->lock_thresh    = lock_thresh;  
  pll->max_lock_value = -100.0f;
  
  pll_filter_init (&(pll->loop_filter),
                   loop_cutoff_hz,
                   sample_rate,
                   one_over_root_two);
  
  pll_filter_init (&(pll->lock_filter),
                   lock_cutoff_hz,
                   sample_rate,
                   one_over_root_two);
  
}

// returns phase of ref_sig
float qb_pll_process (qb_pll_t *pll, double in) { //, qb_fir_t *lp_1k9) {
//float qb_pll_process (qb_pll_t *pll, float in) { //, qb_fir_t *lp_1k9) {

  double t;
  double quad_ref;
  double rate;
  double pll_loop_control;
  double h;
  double ref_phase;
  double period;
  s32_t i;
  
  // lp_1k9 passed in is NULL for the calibration (=> tape speed, not currently used) run.
  // it's needed for computing phase_error.
  
  rate   = pll->sample_rate;
  //period = 1.0 / rate;
  period = 1.0f / rate;
  
  pll_loop_control = in * pll->ref_sig * pll->loop_gain;
  
  h = pll_filter_process (&(pll->loop_filter), pll_loop_control); // still double precision
  pll_loop_control = h;
  pll->pll_integral += (float) (pll_loop_control / rate);
  t = pll->samples_processed / rate;
  ref_phase = pll->pll_cf * (t + pll->pll_integral);
  //pll->ref_sig = cos (2.0 * QB_PI * ref_phase);
  pll->ref_sig = cosf (2.0f * ((float) QB_PI) * (float) ref_phase);
  quad_ref = ((pll->ref_sig - pll->old_ref) * rate) / (2.0 * QB_PI * pll->pll_cf);
  //quad_ref = ((pll->ref_sig - pll->old_ref) * rate) / (2.0f * ((float) QB_PI) * pll->pll_cf);
  pll->debug_quad_ref = quad_ref;
  pll->old_ref = pll->ref_sig;
  pll->pll_lock = pll_filter_process (&(pll->lock_filter), -quad_ref * in);
  (pll->samples_processed)++;
  
  if (pll->pll_lock > pll->lock_thresh) {
    if ( ! pll->locked ) {
      //~ printf("[%u] locked (pll_lock = %lf)\n", n, pll->pll_lock);
      pll->locked = 1;
      pll->num_locks++;
    }
    pll->time_locked += period;
  } else {
    if (pll->locked) {
      //~ printf("[%u] lost lock (pll_lock = %lf)\n\n", n, pll->pll_lock);
      pll->locked = 0;
    }
  }
  
  if (pll->pll_lock > pll->max_lock_value) {
    pll->max_lock_value = pll->pll_lock;
  }
  
  // get integer part of phase
  i = (s32_t) ref_phase;
  // extract fractional part of phase (0.0 -> 1.0)
  return (float) (ref_phase - ((double) i));

}


double qb_pll_get_tape_speed (qb_pll_t *pll) {
  return (1.0f + (pll->pll_integral / pll->time_locked));
}

/*
void qb_pll_populate_inspect_values_slice (qb_pll_t *pll, s16_t pll_inspect[4]) {
  double lock;
  pll_inspect[0] = qb_double_to_s16(pll->ref_sig);
  pll_inspect[1] = qb_double_to_s16(pll->pll_integral);
  lock = pll->pll_lock;
  if (fabs(lock) > 1.0) { // clamp
    lock = (lock >= 0.0) ? 1.0 : -1.0;
  }
  pll_inspect[2] = qb_double_to_s16(lock);
  pll_inspect[3] = qb_double_to_s16(pll->debug_quad_ref * 0.3);
  // pll_inspect[4] is jitter, and we would need more context to write that :(
}
*/
void qb_pll_populate_inspect_values_slice (qb_pll_t *pll, s16_t pll_inspect[4]) {
  double lock;
  pll_inspect[0] = qb_double_to_s16(pll->ref_sig);
  pll_inspect[1] = qb_double_to_s16(pll->pll_integral);
  lock = pll->pll_lock;
  //if (fabsf(lock) > 1.0f) { // clamp
  if (fabs(lock) > 1.0) { // clamp
    //lock = (lock >= 0.0f) ? 1.0f : -1.0f;
    lock = (lock >= 0.0) ? 1.0 : -1.0;
  }
  pll_inspect[2] = qb_double_to_s16(lock);
  //pll_inspect[3] = qb_double_to_s16(pll->debug_quad_ref * 0.3f);
  pll_inspect[3] = qb_double_to_s16(pll->debug_quad_ref * 0.3);
  // pll_inspect[4] is jitter, and we would need more context to write that :(
}


static void pll_filter_init (qb_pll_filter_t *filter,
                             double frequency,
                             u32_t sample_rate,
                             double q_factor) {

  double omega;
  double sine, cosine;
  double alpha;
  
  memset(filter, 0, sizeof(qb_pll_filter_t));
  
  // ???
  // "only used for peaking and shelving filter types"
  
  omega = (2.0 * QB_PI * frequency) / (double) sample_rate;
  
  cosine = cos(omega);
  sine = sin(omega);
  
  alpha = sine / (2.0 * q_factor);

  // -- not used for lowpass ?? --
  //~ gain = pow(10, gain_db / 40.0);
  //~ beta = sqrt(2.0 * gain);
  
  // low pass only stuff
  filter->b0 = (1.0 - cosine) / 2.0;
  filter->b1 = 1.0 - cosine;
  filter->b2 = (1.0 - cosine) / 2.0;
  filter->a0 = 1.0 + alpha;
  filter->a1 = -2.0 * cosine;
  filter->a2 = 1.0 - alpha;
  
  filter->b0 /= filter->a0;
  filter->b1 /= filter->a0;
  filter->b2 /= filter->a0;
  filter->a1 /= filter->a0;
  filter->a2 /= filter->a0;
  
}

static double pll_filter_process (qb_pll_filter_t *f, double in) {

  double out;
  
  out =   (f->b0 * in)
        + (f->b1 * f->x1)
        + (f->b2 * f->x2)
        - ((f->a1 * f->y1) + (f->a2 * f->y2));
    
  // propagate
  
  f->x2 = f->x1;
  f->x1 = in;
  
  f->y2 = f->y1;
  f->y1 = out;
  
  return out;
   
}
