/*
 *  Quadbike 2
 *  Copyright (C) 2026 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "csw.h"
#include "qbio.h"

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>

#define CSW_ALLOC_DELTA 1000

static qb_err_t csw_append_run (u8_t **csw,
                                size_t *alloc,
                                size_t *fill,
                                s64_t run_len);
                            
static qb_err_t csw_append_bytes (u8_t **csw_inout,
                                  size_t *csw_alloc,
                                  size_t *csw_fill,
                                  char *in,
                                  s64_t in_len64);

#define QB_CSW_HEADER_LEN 52

qb_err_t qb_write_csw  (char *path,
                        qb_span_t *spans,
                        s32_t num_spans,
                        s32_t sample_rate,
                        s64_t input_len_smps,
                        u8_t allow_overwrite) {
                        //u8_t display_progress) {
                     
  u8_t *csw_body, *csw, *zbody;
  size_t body_alloc;
  size_t body_fill;
  size_t zbody_fill;
  u32_t rate32u;
  char rate32_s[4];
  u32_t num_pulses;
  char num_pulses_s[4];
  char flags;
  char app_description[16];
  size_t app_description_len;
  char *version;
  s32_t n;
  size_t pulses_ix;
  qb_err_t e;
  s64_t total_samples;
  char hdr[QB_CSW_HEADER_LEN]; // 52
  char *hdrp;
  u8_t csw_type;
  s32_t sn;
  s32_t total_cycles;
  qb_atom_t *prev_cyc;
  s64_t singles, pairs;
  
  e = QB_E_OK;
  
  csw_body = NULL;
  zbody = NULL;
  zbody_fill = 0;
  body_alloc = 0;
  body_fill = 0;
  flags = 0;
  memset(app_description, 0, 16);
  num_pulses = 0;
  total_cycles = 0;
  
  rate32u = 0x7fffffff & sample_rate;
  qb_u32_to_little_endian (rate32u, rate32_s);
  
  flags |= 1;

  version = QB_VERSION_STRING (QB_VERSION_U32_MAJOR, QB_VERSION_U32_MINOR, QB_VERSION_U32_SUB);
  app_description_len = strlen(version);
  if (app_description_len > 16) {
    fprintf(QB_ERR, "B: %s: app description length exceeds allowed space in CSW header\n", QB_FUNC_M);
    return QB_E_BUG;
  }
  memcpy (app_description, version, app_description_len);

  // generate CSW header
  
  hdrp = hdr;
  
  memcpy(hdrp, "Compressed Square Wave", 22);
  hdrp += 22;
  memcpy(hdrp, "\x1a\x02\x00", 3);
  hdrp += 3;
  memcpy(hdrp, rate32_s, 4);
  hdrp += 4;
  pulses_ix = hdrp - hdr;
  memcpy(hdrp, "NPLS", 4); // placeholder for total num pulses
  hdrp += 4;
#ifdef QB_HAVE_ZLIB
  *hdrp = 2;     // compression type 2: RLE + zlib
#else
  *hdrp = 1;     // compression type 1: RLE
#endif
  hdrp++;
  *hdrp = flags; // only bit 0: initial polarity
  hdrp++;
  *hdrp = 0;     // header extension length 0
  hdrp++;
  memcpy(hdrp, app_description, 16);

  total_samples  = 0;
  prev_cyc       = NULL;
  singles        = 0;
  pairs          = 0;
  
  for (sn=0; sn < num_spans; sn++) {
  
    qb_span_t *span;
    s32_t num_cycles;
    qb_atom_t *cycles;
    qb_atom_t *cyc;
  
    cyc  = NULL;
    span = spans + sn;
  
    if (span->num_atoms < 1) {
      fprintf(QB_ERR, "B: %s [%lld]: span %d (%s): bad cycle count (%d)\n",
              QB_FUNC_M, qb_get_span_start_accurate_or_rough(span), sn, qb_span_get_type_text(span), span->num_atoms);
      return QB_E_BUG;
    }
    
    num_cycles   = span->num_atoms;
    cycles       = span->atoms;
  
    // data starts here
    for (n=0; n < num_cycles; n++, total_cycles++) {
    
      s64_t len_samples;
      u8_t v;
      s64_t sn0, sn1;
      
      cyc = cycles + n;
      
      /*
      // hax0r the first cycle so that it starts at zero
      if ((0==sn) && (0==n)) {
        if (cyc->sample_num > 0) {
          fprintf(QB_ERR, "  W: nudging first cycle: [%lld] -> [0]\n",
                  cyc->sample_num);
          cyc->sample_num = 0;
        }
        prev_cyc = cyc;
        continue;
      }
      */
      
      // silent spans now have 2-3 cycles; a default two to preserve the polarity,
      // and then possibly a third to invert it
      if ((QB_SPAN_TYPE_SILENT == span->type) && (span->num_atoms > 3)) {
        fprintf(QB_ERR, "E: span %d (start %lld) is silent, but has excessive num cycs (has %d)\n",
                sn, span->start, span->num_atoms);
        e = QB_E_BUG;
        break;
      }
      
      if ((QB_SPAN_TYPE_SILENT == span->type) && (span->num_atoms < 2)) {
        fprintf(QB_ERR, "E: span %d (start %lld) is silent, but has too few cycs (has %d)\n",
                sn, span->start, span->num_atoms);
        e = QB_E_BUG;
        break;
      }
      
      // skip very first cycle
      if ((0==n) && (0==sn)) {
        prev_cyc = cyc;
        continue;
      }

      sn0 = prev_cyc->sample_num;
      sn1 = cyc->sample_num;
      
      len_samples = (sn1 - sn0);
      
      if (len_samples < 1) {
        fprintf(QB_ERR, "E: [%lld] span %d (%s, start %lld, len %lld): adjacent cycs w/same smpnum :/\n",
                sn0,
                sn,
                qb_span_get_type_text(span),
                qb_get_span_start_accurate_or_rough(span),
                span->len);
        e = QB_E_CSW_ZERO_OR_NEG_RUN_LEN;
        break;
      }
    
//#ifdef QB_SANITY
      //if (len_samples != cyc->samples_since_last) {
      //  fprintf(QB_ERR, "E: span %d (%s, span start %lld, smpnum %lld): bad samples_since_last value on cycle %d (smplen %lld, SSL %lld)\n",
      //          sn, qb_span_get_type_text(span), span->start, span->atoms[0].sample_num, n, len_samples, cyc->samples_since_last);
      //  e = QB_E_BUG;
      //  break;
      //}
//#endif
      
      // len_samples has to be 2 or more, because it's impossible to create
      // a pair of pulses (high, low) to fit into a single sample!
      
  //#ifdef QB_SANITY
      if (len_samples < 2) {
        fprintf(QB_ERR, "B: %s: span %d: cyclen < 2 (cycle %d; smps [%lld, %lld])\n", QB_FUNC_M, sn, n, sn0, sn1);
        e = QB_E_CSW_ZERO_OR_NEG_RUN_LEN;
        break;
      }
 // #endif
    
      if (len_samples & 0xffffffff00000000) {
        fprintf(QB_ERR, "B: %s: span %d: len_samples would overflow u32!\n", QB_FUNC_M, sn);
        e = QB_E_CSW_LEN_SAMPLES_OVERFLOW;
        break;
      }

      v = prev_cyc->value;
      
#ifdef QB_SANITY
      if ( ('0' != v) && ('1' != v) ) {
        fprintf(QB_ERR, "B: %s: [%lld]: bad cyctype %c, span %d (%s): S and L were abolished!\n",
                QB_FUNC_M, sn0, v, sn, qb_span_get_type_text(span));
        e = QB_E_BUG;
        break;
      }
#endif
      
      if ('0' == v) {
        // write HALF cycle of 1200 Hz tone (or a long silent one)
        e = csw_append_run (&csw_body,
                            &body_alloc,
                            &body_fill,
                            0xffffffff & len_samples);
        num_pulses++;
      } else if ('1' == v) {
        // write FULL cycle of 2400 Hz tone
        e = csw_append_run (&csw_body,
                            &body_alloc,
                            &body_fill,
                            0xffffffff & (len_samples / 2));
        if (QB_E_OK != e) { break; }
        e = csw_append_run (&csw_body,
                            &body_alloc,
                            &body_fill,
                            0xffffffff & (len_samples - (len_samples / 2)));
        num_pulses+=2;
      } else {
        // defend
        fprintf(QB_ERR, "B: %s: unknown cycle value %c\n", QB_FUNC_M, v);
        e = QB_E_BUG;
      }
      
      if (QB_E_OK != e) { break; }
      
      // update stats
      if ('0' == v) {
        singles++;
      } else {
        pairs++;
      }
      total_samples += len_samples;
      
      prev_cyc = cyc;
      
      if (QB_E_OK != e) { break; }

    } // next cycle
    
    //if (QB_E_CSW_ZERO_OR_NEG_RUN_LEN == e) {
    //  fprintf(QB_ERR, "B: [%lld]: qb_write_csw: zero/negative run length: cycs: (%lld, %lld)\n",
    //          total_samples, prev_cyc->sample_num, cyc->sample_num);
    //}
    
    if (QB_E_OK != e) {
      fprintf(QB_ERR, "E: [%lld] breaking on %s span %d due to error %u\n",
              qb_get_span_start_accurate_or_rough(span),
              qb_span_get_type_text(span),
              sn,
              e);
      break;
    }
    
  } // next span
  
  // final piece
  if (QB_E_OK == e) {
    e = csw_append_run (&csw_body,
                        &body_alloc,
                        &body_fill,
                        0xffffffff & (input_len_smps - prev_cyc->sample_num));
  }
  num_pulses++;
  
  if (QB_E_OK != e) {
    qb_free(csw_body);
    return e;
  }
    
  printf("CSW: singles %lld; pairs %lld; pulses %u; samples %lld.\n",
         singles, pairs, num_pulses, total_samples);
  fflush(stdout);
  
  qb_u32_to_little_endian (num_pulses, num_pulses_s);
  
  // replace placeholder num pulses value in header
  memcpy(hdr + pulses_ix, num_pulses_s, 4);
  
#ifdef QB_HAVE_ZLIB
  printf("Compressing CSW body: %zu -> ", body_fill);
  fflush(stdout);
  e = qb_zlib_compress (csw_body, body_fill, 0, &zbody, &zbody_fill); // 0 = don't use gzip encoding
  qb_free(csw_body);
  if (QB_E_OK != e) { return e; }
  printf("%zu bytes.\n", zbody_fill);
#else
  // just steal the pointer
  zbody      = csw_body;
  zbody_fill = body_fill;
#endif
  
  csw = qb_malloc (QB_CSW_HEADER_LEN + zbody_fill);
  if (NULL == csw) {
    fprintf(QB_ERR, "E: Out of memory allocating CSW file buffer.\n");
    qb_free(zbody);
    return QB_E_MALLOC;
  }
  
  memcpy(csw, hdr, QB_CSW_HEADER_LEN);
  memcpy(csw + QB_CSW_HEADER_LEN, zbody, zbody_fill);

  qb_free(zbody);

  e = qb_write_file (path,
                     allow_overwrite,
                     csw,
                     QB_CSW_HEADER_LEN + zbody_fill);
  if (QB_E_OK == e) {
#ifdef QB_HAVE_ZLIB
    csw_type = 2;
#else
    csw_type = 1;
#endif
    printf("Wrote type %u CSW: %s\n", csw_type, path);
  }
  
  qb_free(csw);
  
  return e;
  
}




static qb_err_t csw_append_run (u8_t **csw,
                                size_t *alloc,
                                size_t *fill,
                                s64_t run_len64) {
  
  u32_t run_len;
  char run_len_s[5] = { 0 };

  if (run_len64 < 1) {
    fprintf(QB_ERR, "E: csw_append_run: zero or negative run_len (%lld)\n", run_len64);
    return QB_E_CSW_ZERO_OR_NEG_RUN_LEN;
  }

  if (run_len64 > 0xffffffff) {
    fprintf(QB_ERR, "E: csw_append_run: run_len would overflow u32 (%lld)\n", run_len64);
    return QB_E_CSW_RUN_LEN_OVERFLOW;
  }

  run_len = 0xffffffff & run_len64;
                            
  if (run_len > 255) {
  
    qb_u32_to_little_endian (run_len, run_len_s + 1);
    csw_append_bytes(csw, alloc, fill, run_len_s, 5);
    
  } else {
  
    run_len_s[0] = 0xff & run_len;
    csw_append_bytes(csw, alloc, fill, run_len_s, 1);
               
  }
  
  return QB_E_OK;
  
}


static qb_err_t csw_append_bytes (u8_t **csw_inout,
                                  size_t *csw_alloc,
                                  size_t *csw_fill,
                                  char *in,
                                  s64_t in_len64) {
  u8_t *csw_new;
  //u32_t in_len;
  if (in_len64 < 0) {
    fprintf(QB_ERR, "B: %s: negative in_len (%lld)\n", QB_FUNC_M, in_len64);
    return QB_E_BUG;
  }
  if (in_len64 > 0xffffffff) {
    fprintf(QB_ERR, "B: %s: in_len would overflow u32 (%lld)\n", QB_FUNC_M, in_len64);
    return QB_E_CSW_RUN_LEN_OVERFLOW;
  }
  in_len64 = 0xffffffff & in_len64;
  if (*csw_fill + in_len64 >= *csw_alloc) {
    *csw_alloc += in_len64 + CSW_ALLOC_DELTA;
    csw_new = qb_realloc(*csw_inout, *csw_alloc);
    if (NULL == csw_new) {
      fprintf(QB_ERR, "E: Out of memory appending CSW bytes.\n");
      return QB_E_MALLOC;
    }
    *csw_inout = csw_new;
  }
  memcpy(*csw_inout + *csw_fill, in, in_len64);
  *csw_fill += in_len64;
  return QB_E_OK;
}

