/*
 *  Quadbike 2
 *  Copyright (C) 2026 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "qbio.h"
#include "audio.h"
#include "util.h"

#include <stdio.h>
#ifndef QB_WINDOWS
#include <unistd.h>
#endif
#include <string.h>
#include <stdlib.h>

#include <sys/stat.h>
#include <sys/types.h>

/*
qb_err_t qb_load (char *filename,
                  u8_t *buf,
                  s32_t bufsize,
                  s32_t *actual_len_out) {

  FILE *f;
  qb_err_t e;
  
  f = qb_fopen (filename, "rb");
  
  if (NULL == f) {
    fprintf(QB_ERR, "Failed to open file %s\n", filename);
    return QB_E_FOPEN;
  }
    
  e = qb_fread_fully (buf, bufsize, f);

  *actual_len_out = (s32_t) (0x7fffffff & ftell(f));

  if (QB_E_FREAD_EOF == e) {
    // ignore EOF
    e = QB_E_OK;
  }
  
  fclose(f);
  
  return e;
  
}
*/


FILE *qb_fopen (char *filename, char *mode) {
  
  FILE *f = NULL;
  u8_t failed=0;
#ifdef QB_WINDOWS
  int winfnlen;
  LPWSTR winfn;
  int winmodelen;
  LPWSTR winmode;
  qb_err_t e;
#endif
  
#ifdef QB_SANITY
  if (!filename) {
    fprintf(QB_ERR, "B: %s: filename==NULL\n", QB_FUNC_M);
    return NULL;
  }
  if (!mode) {
    fprintf(QB_ERR, "B: %s: mode==NULL\n", QB_FUNC_M);
    return NULL;
  }
#endif

#ifdef QB_WINDOWS
  /*
  winfnlen = MultiByteToWideChar (CP_UTF8, MB_ERR_INVALID_CHARS, filename, -1, NULL, 0);
  winfn = qb_malloc(winfnlen * sizeof(wchar_t));
  if (NULL == winfn) {
    fprintf(QB_ERR, "E: Out of memory allocating file name buffer for fopen.\n");
    return NULL;
  }
  MultiByteToWideChar (CP_UTF8, MB_ERR_INVALID_CHARS, filename, -1, winfn, winfnlen);
  */

  e = qb_windows_convert_utf8_string_to_wchar (filename, &winfn, &winfnlen);
  if (QB_E_OK != e) { return NULL; }

  /*
  winmodelen = MultiByteToWideChar(CP_UTF8, MB_ERR_INVALID_CHARS, mode, -1, NULL, 0);
  winmode = qb_malloc(winmodelen * sizeof(wchar_t));
  if (NULL == winmode) {
    fprintf(QB_ERR, "E: Out of memory allocating wide mode buffer for fopen.\n");
    qb_free(winfn);
    return NULL;
  }
  MultiByteToWideChar(CP_UTF8, MB_ERR_INVALID_CHARS, mode, -1, winmode, winmodelen);
  */

  e = qb_windows_convert_utf8_string_to_wchar (mode, &winmode, &winmodelen);
  if (QB_E_OK != e) { failed = 1; }

  if ( ( ! failed ) && _wfopen_s (&f, winfn, winmode)) {
    failed = 1;
  }

  qb_free(winfn);
  qb_free(winmode);

#else
  if (NULL == (f = fopen (filename, mode))) {
    failed = 1;
  }
#endif
  if (failed) {
    fprintf(QB_ERR, "E: Could not open file \"%s\".\n", filename); //, errmsg);
  }
  return f;
}


qb_err_t qb_read_file (char *filename,
                       u8_t **payload_out,
                       s64_t *len_out,
                       s64_t max_size) {
  // *payload_out must be freed by the caller if the result is QB_E_OK
  
  qb_stat_t sb;
  
  qb_err_t ecode;
  FILE *fp;
  u64_t max_size_u;
  
  *payload_out = NULL;
  *len_out = 0;
  ecode = QB_E_OK;
  fp = NULL;
  
#ifdef QB_SANITY
  if (max_size < 0) {
    fprintf(QB_ERR, "B: %s: max_size < 0\n", QB_FUNC_M);
    return QB_E_BUG;
  }
  if (NULL == filename) {
    fprintf(QB_ERR, "B: %s: filename==NULL\n", QB_FUNC_M);
    return QB_E_BUG;
  }
  if (NULL == payload_out) {
    fprintf(QB_ERR, "B: %s: payload_out==NULL\n", QB_FUNC_M);
    return QB_E_BUG;
  }
  if (NULL == len_out) {
    fprintf(QB_ERR, "B: %s: len_out==NULL\n", QB_FUNC_M);
    return QB_E_BUG;
  }
#endif

  max_size_u = max_size & 0x7fffffffffffffff;

  // stat the file once before opening it,
  // to check it isn't a directory or something
  memset(&sb, 0, sizeof(qb_stat_t));
  if (QB_E_OK != (ecode = qb_stat_file (filename, &sb))) {
    fprintf(QB_ERR, "E: File to be loaded is missing:\n   %s\n", filename);
    return ecode;
  }

  if (sb.is_dir) {
    fprintf(QB_ERR, "E: Could not load file, because it is "
              "actually a directory:\n  %s\n", filename);
    return QB_E_FILE_NOT_FILE;
  }
  
  // then, open it
  if (NULL == (fp = qb_fopen (filename, "rb"))) {
    return QB_E_FOPEN_READ_FILE;
  }

  // and now we stat it again
  do { // try {
  
    memset(&sb, 0, sizeof(qb_stat_t));
    
    if (QB_E_OK != (ecode = qb_stat_file(filename, &sb))) {
      break;
    }

    if (sb.is_dir) {
      fprintf(QB_ERR, "E: Could not load file \"%s\", because it is "
                      "actually a directory.\n", filename);
      ecode = QB_E_FILE_NOT_FILE;
      break;
    }
    
    if (0 == sb.fsize) {
      fclose(fp);
      return QB_E_OK;
    }
    
    if (sb.fsize > max_size_u) {
      fprintf(QB_ERR,
              "E: File \"%s\" was too large to open: maximum size %lld"
              ", file size %llu"
              "\n",
              filename,
              max_size,
              sb.fsize);
      ecode = QB_E_FILE_TOOLARGE;
      break;
    }

    //~ if (sizeof(size_t) == 4)
      //~ mask = 0x7fffffff;
    //~ else
      //~ mask = 0x7fffffffffffffff;

  //printf("qb_read_file: %s, allocing %" FMT_SIZET " bytes\n", filename, sb.fsize);

    //~ if (sizeof(sb.fsize) == 4) {
      //~ fprintf(QB_ERR, "E: File \"%s\" was too large to open: maximum size %" FMT_S64 ", file size %" FMT_S64 "\n",
                //~ filename, mask-1, (s64_t) sb.fsize);
      //~ fclose(fp);
      //~ return QB_E_FILE_TOOLARGE;
    //~ } else {
      //~ fprintf(QB_ERR, "E: File \"%s\" was too large to open: maximum size %" FMT_S64 ", file size %" FMT_S64 "\n",
                //~ filename, mask-1, (s64_t) sb.fsize);
      //~ fclose(fp);
      //~ return QB_E_FILE_TOOLARGE;
    //~ }

    *payload_out = qb_malloc (sb.fsize + 1);
    if (NULL == *payload_out) {
      fprintf(QB_ERR,
              "E: Could not allocate %llu bytes for input file \"%s\".\n",
              sb.fsize+1,
              filename);
      ecode = QB_E_MALLOC;
      break;
    }
    
    (*payload_out)[sb.fsize] = 0; // allocate an extra byte at the end of the block and set it to 0
    
  } while (0);
  
  if (QB_E_OK != ecode) { // catch {
    fclose(fp);
    return ecode;
  }

  //if (sizeof(size_t) == 4)
  //  *payload_out = QB_qb_malloc (sb.fsize & 0x7fffffff, "qb_read_file payload");
  //else
  //  *payload_out = QB_qb_malloc (sb.fsize & 0x7fffffffffffffff, "qb_read_file payload");

  //~ if (sizeof(off_t) == 4 || sizeof(size_t) == 4)
    //~ mask = 0x7fffffff;
  //~ else
    //~ mask = 0x7fffffffffffffff;

  ecode = qb_fread_fully (*payload_out, sb.fsize, fp);
  fclose(fp);
  
  if (QB_E_OK != ecode) {
    qb_free(*payload_out);
    *payload_out = NULL;
    return ecode;
  }
  
  *len_out = sb.fsize & 0x7fffffffffffffff; // & ((size_t) mask);
  
//printf("qb_read_file(%s) returns\n", filename);

  return ecode;
  
}


// FIXME: pass in filename for errmsg
qb_err_t qb_fread_fully (u8_t *buf, size_t size, FILE *f) {
  size_t size_read=0;
  //~ char *errmsg = NULL;
  while (size_read<size) {
    size_t j;
    size_read += (j = fread (buf+size_read, 1, size-size_read, f));
    if (ferror(f)) {
      //~ errmsg = QB_STRERROR();
      fprintf(QB_ERR, "E: Problem reading file.\n"); //, errmsg);
      return QB_E_FREAD;
    } else if (feof(f)) {
      return QB_E_FREAD_EOF;
    }
    if (!j) {
#ifndef QB_WINDOWS
      usleep(1);
#else
      Sleep(1);
#endif
    }
  }
  return QB_E_OK;
}



// FIXME: add follow_link parameter, choose between stat() and lstat()
qb_err_t qb_stat_file (char *filename, qb_stat_t *stat_out) {

#ifdef QB_WINDOWS
  WIN32_FILE_ATTRIBUTE_DATA fad;
  u64_t tmpu64;
  int winfnlen;
  LPWSTR winfn;
  u8_t b;
#else
  struct stat sb;
#endif

  if ( NULL == filename ) {
    fprintf(QB_ERR, "B: %s: filename==NULL\n", QB_FUNC_M);
    return QB_E_BUG;
  }
  if ( NULL == stat_out ) {
    fprintf(QB_ERR, "B: %s: stat_out==NULL\n", QB_FUNC_M);
    return QB_E_BUG;
  }

#ifdef QB_WINDOWS
  memset(&fad, 0, sizeof(WIN32_FILE_ATTRIBUTE_DATA));
#else
  memset(&sb, 0, sizeof(struct stat));
#endif

  memset(stat_out, 0, sizeof(qb_stat_t));

#ifdef QB_WINDOWS

  winfnlen = MultiByteToWideChar(CP_UTF8, 0, filename, -1, NULL, 0);
  winfn = qb_malloc(winfnlen * sizeof(wchar_t));

  if (NULL == winfn) {
    fprintf(QB_ERR, "E: Out of memory allocating file stat buffer.\n");
    return QB_E_MALLOC;
  }

  MultiByteToWideChar(CP_UTF8, 0, filename, -1, winfn, winfnlen);

  b = GetFileAttributesExW(winfn, GetFileExInfoStandard, &fad);
  qb_free(winfn);

  if ( ! b ) {
    return QB_E_STAT;
  }

//printf("file attributes(%s) = 0x%x\n", filename, fad.dwFileAttributes);
  if (fad.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
//printf("is_dir=1\n");
    stat_out->is_dir = 1;
  }
  tmpu64 = fad.nFileSizeHigh;
  tmpu64 <<= 32;
  tmpu64 &= 0xffffffff00000000UL;
  tmpu64 |= 0xffffffff & fad.nFileSizeLow;
  stat_out->fsize = (off_t) 0x7fffffffffffffff & tmpu64;
#else // ndef QB_WINDOWS
  if (lstat(filename, &sb)) {
    return QB_E_STAT;
  }
  if ((sb.st_mode & S_IFMT) == S_IFLNK) {
    stat_out->is_link = 1;
  }
  if (stat(filename, &sb)) {
    return QB_E_STAT;
  }
  if ((sb.st_mode & S_IFMT) == S_IFDIR) {
    stat_out->is_dir = 1;
  }

  stat_out->fsize = sb.st_size;
#endif
  return QB_E_OK;
}


qb_err_t qb_check_can_write_file (char *filename, u8_t overwrite) {

  qb_stat_t sb;
  char *symlink_text = " ";
  
//~ printf("qb_check_can_write_file (\"%s\", overwrite=%u)\n", filename, overwrite);

  if (NULL == filename) {
    fprintf(QB_ERR, "B: %s: filename is NULL!\n", QB_FUNC_M);
    return QB_E_BUG;
  }

  // does the file exist? if so, do we have permission to overwrite it?
  memset(&sb, 0, sizeof(qb_stat_t));
  if (QB_E_OK == qb_stat_file(filename, &sb)) {
    // stat succeeded: file exists
    if (overwrite) {
      // but we are cleared to overwrite it
      if (sb.is_dir || sb.is_link) {
        // but we can't, because it's a directory or symlink
#ifndef QB_WINDOWS
        symlink_text = " a symbolic link or ";
#endif
        fprintf(QB_ERR, "E: Could not overwrite existing file because it is "
                 "not actually a file.\nE: It is"
                 "%s"
                 "a directory:\nE:   %s\n",
                 symlink_text,
                 filename);
        return QB_E_FILE_NOT_FILE;
      }
    } else {
      fprintf(QB_ERR, "E: Refusing to overwrite file \"%s\" (try %s).\n", filename, QB_CLI_S_OVERWRITE);
      return QB_E_FILE_EXISTS;
    }
  }
  
  return QB_E_OK;
  
}



qb_err_t qb_write_file (char *filename, u8_t overwrite,
                        u8_t *payload, size_t len) {

  FILE *fp = NULL;
  qb_err_t e = QB_E_OK;
    
  // general file writing scheme

  e = qb_check_can_write_file (filename, overwrite);
  if (QB_E_OK != e) { return e; }
  
  //~ if (mkpath) {
    //~ if (QB_E_OK != (ecode = mkdir_for_file (filename))) {
      //~ return ecode;
    //~ }
  //~ }
  
  // write the payload
  if (NULL == (fp = qb_fopen (filename, "wb"))) {
    fprintf(QB_ERR, "E: Could not open file for writing: \"%s\"\n", filename);
    return QB_E_FOPEN_OPFILE;
  }
  
  // prevent time-of-check to time-of-use bug
  // however we must call with overwrite=1, since
  // the file now exists; we just created it!
  e = qb_check_can_write_file (filename, 1); 
  if (QB_E_OK != e) {
    fclose(fp);
    return e;
  }
  
  e = qb_fwrite_fully (payload, len, filename, fp);
  fclose(fp);
  return e;
}


#include <errno.h>

qb_err_t qb_fwrite_fully (u8_t *buf,
                          size_t size,
                          char *filename,
                          FILE *f) {
  size_t size_written=0;
  //~ char *errmsg = NULL;
  
  if (NULL == f) {
    fprintf(QB_ERR, "B: %s: f is NULL\n", QB_FUNC_M);
    return QB_E_BUG;
  }
  
  while (size_written < size) {
    size_t j;
//printf("buf=%p\n", buf);
//~ printf("fwrite(%p, 1, %zu, [%s])\n", buf+size_written, size-size_written, filename_for_printing);

//qb_update_meter (size_written, size, 0.0f, 0);

    size_written+=(j=fwrite(buf+size_written,1,size-size_written,f));
    
    if (ferror(f)) {
      //~ errmsg = QB_STRERROR();
      fprintf(QB_ERR, "E: Error writing file: %s\n", filename);
//~ printf("%s\n", strerror(errno));
      return QB_E_FWRITE;
    } else if (feof(f)) {
      //~ errmsg = QB_STRERROR();
      fprintf(QB_ERR, "E: EOF writing file: %s\n", filename);
      return QB_E_FREAD_EOF;
    }
    if (!j) {
#ifndef QB_WINDOWS
      usleep(1);
#else
      Sleep(1);
#endif
    }
  }
  return QB_E_OK;
}








//~ // pass in a filename and it will make a directory
//~ // to contain that file
//~ qb_err_t mkdir_for_file (char *filename) {
  //~ char *dirname = NULL;
  //~ qb_err_t ecode = QB_E_OK;
  //~ // strip filename off path
  //~ ecode = qb_getpath (filename, &dirname);
//~ //printf("[%u] %s -> %s\n", ecode, filename, dirname);
  //~ switch (ecode) {
    //~ case QB_E_OK:
      //~ // have path, make directory
      //~ ecode = qb_mkdir (dirname);
      //~ qb_free(dirname, "getpath buf");
      //~ break;
    //~ case QB_E_GETPATH_NOSEP:
      //~ // no path, just filename, no mkdir needed
      //~ ecode = QB_E_OK;
      //~ break;
    //~ default:
      //~ // error
      //~ return ecode;
  //~ }
  //~ return ecode;
//~ }


 
