/*
 *  Quadbike 2
 *  Copyright (C) 2026 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "quadbike.h"
#include "fir.h"
#include "pll.h"
#include "util.h"
#include "qbio.h"
#include "goertzel.h"
#include "csw.h"
#include "process.h"
#include "errcorr.h" // for QB_ERROR_CORRECTION_...
#include "cli_stgs.h"
#include "inspect.h"
#include "sync_pll.h"

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

// [ ] bug? if final silent span is extremely short, placing a /time hint before it
//     will cause fail on loading by the TIBET reference implementation, because said silence is obliterated by the loader,
//     then it breaks the "no dangling hints" condition
//     not sure if this is a quadbike problem or a loader problem
//     (you know, actually it's probably a loader problem)


// TEST PROBLEMS
// [ ] trunc-test.txt

// PHP
// test csw2wav.php on hardware
// tibetuef.php:
// - GZIP in
// - GZIP out

// TESTS
// [x] convert entire corpus to 22050 and 48000 and test entire corpus at both rates
// [x] confirm every command line option
// [ ] short input, truncated input
// [ ] test build with various #ifdef combinations
// [x] valgrind leaks etc., w/all -s modes, and w/--inspect_dir (just run all the tests under VG)
// [x] filename collisions, including fully-qualified and relative paths
// [x] bonkers path like "////////", although "///////a" should be allowed
// [x] testing colliding filenames: MacOS + win32 are case sensitive, so /users/ali/lol should fail against /Users/ali/lol
// [ ] windows: test path names longer than 0x7fff characters
// [ ] Centipede_(Superior_Software)_SideCornerRivets.flac is giving negative sample num on "Removing empty data span"

// [ ] atom
// [ ] include TIBET ref decoder
// [x] add --silence-thresh
// [ ] --phase-det pll could be SIMDed. The four carriers would need to be swizzled into
//     a buffer of four-element vectors, and then four PLLs could be run on them at the same time.
//     the swizzling could be achieved for nothing, since the carriers come out of the filter
//     in a different swizzled form, so rather than being unswizzled back into four linear
//     buffers, they could be unswizzled out into one buffer of four-element vectors for free.
// [ ] concurrency? particularly for swizzling and unswizzling?
// [x] BUG: double silent span at start:
//     ./quadbike -s freq -! -t /tmp/qb.tibet /Users/ali/tapes/Centipede_(Superior_Software)_SideCornerRivets.wav
//     -- fixed, added extra check and -v warning for data spans converted to silent if empty;
//        hopefully resolved by adding another call to amalgamate_spans()
// [x] bolt the TIBET reference decoder onto the unit tests, so that all TIBETs are run through it;
// [x] **** update docs to make it clear you have to run it from cmd.exe .....
// [ ] --- also figure out how to throw up a dialogue box on windows saying LOL if no CLI args
// [x] Embarrassingly, quadbike can produce illegal TIBETs
//     from the tape framing tests (csw->wav->TIBET):
//       TIBET: line 111: silence is too short: "0.000023"
// [x] stick zlib support in tibetuef
// [ ] find out whereabouts in Estra and Fortress the parity switching occurs, so
//     we can do a test of split span with framing hint
// [x] can't build without QB_ZLIB or whatever, fix this
// [x] --hush-thresh
// [X] DOCS: also hush sum signal->silent conversion only operates on data spans now, not leader
// [x] unit test for --hush-thresh
// [ ] add concat TIBET testing to unit tests if not already there
// [x] test concat TIBET with tibetuef.php
// [ ] note in docs that csw2wav.php has been updated to support "v2.1" CSW w/weirdo flags
// [x] make sure cswblks.php can load the above "v2.1" CSWs too
// [ ] DOCS: updated cswblks.php logic; better resync / error-handling behaviour;
//     blocks > 256 bytes now have block len high byte zeroed (Ultron again)

// [ ] --scan, search for optimal --scale-hf-pwr setting
// [ ] UEF?
// [ ] eliminate silent cycles, leader cycles ('S', 'L')
//     -- there should only be two types of cycle, a singlepulse ('0'), and a doublepulse ('1')
//     -- hmm, dunno? bring this back?
// [ ] 'Invert-o-spans' output needs multiline formatting -- does this even still exist? (-v maybe?)
// [ ] write another inspect file showing -e bitflips.
// [ ] if using --phase-det walk and -p auto but with -s sync, then it should be possible to
//     detect the walk phase FIRST, and then generate ONLY the carrier needed to satisfy
//     the phase detected -- at the moment it generates all four carriers unnecessarily
// [ ] rename cycles to atoms in code: cycle.c/.h, qb_atom_t, etc.
// [ ] 300 baud (odd-run checks should be extended to not-multiple-of-8 checks)
// [ ] acorn atom
// [ ] experiment: always decode at phase 0, but have a front-end differentiator
//     which repeatedly differentiates each span until phase 0 is detected?
// [ ] (option to) write power0 and power1 inspect files as a stereo WAV?
// [ ] --trim option? (just throw out 'S' values?)
// [ ] lowpass filter the power0 and power1 traces

// ignore this
#define BEHIND_A_BLIND_OF_PLASTIC_PLANTS

static qb_err_t qb_cli (qb_t *qb, int argc, char *argv[]);
static void qb_finish (qb_t *qb);
static void print_dupe_cli_error (char *swch);
static void qb_usage (char *argv0) ;
static qb_err_t qb_load_ipfile (qb_t *qb);
static qb_err_t parse_source_selector_simple (char *argv, qb_t *qb);
static const char *phase_scheme_to_string (u8_t ps);
static void print_cli_summary (qb_t *qb);
static qb_err_t main_prologue(int *argc, char *argv[], char ***my_argv_out);
static qb_err_t cli_parse_decimal (char *s, u8_t allow_negative, double *result);
static qb_err_t check_all_filenames_are_sane (qb_t *qb);
static qb_err_t cli_parse_int (char *s, u8_t allow_negative, u8_t max_string_len, s64_t *result);

#define QB_OPFILE_DEFAULT               "qb.tibet"

// #define QB_SCHMITT_TRIGGER_THRESH_OPTIONS

static void qb_usage (char *argv0) {

  // space sections out?
  // if this is preferred, set this to "\n", else ""
  char *spacer = "\n";

  //printf("%sUsage:\n", spacer);
  printf("Usage:\n\n");
  printf("  %s [options] <input file>\n\n", qb_util_basename(argv0));
  printf("Input file is audio supported by libsndfile (e.g. WAV, AIFF, maybe FLAC).\n");
  printf("By default, output is written to %s.\n", QB_OPFILE_DEFAULT); // note added in 2.0.3

  printf("%sOutput options:\n", spacer);
  printf("  %s                    allow overwriting of output files (incl. inspect files)\n",
         QB_CLI_S_OVERWRITE);
  printf("  %s <filename>         output TIBET filename (default: %s)\n",
         QB_CLI_S_OPFILE_TIBET, QB_OPFILE_DEFAULT);
#ifdef QB_HAVE_ZLIB
  printf("  %s <filename>         output TIBETZ (gzipped TIBET) filename\n",
         QB_CLI_S_OPFILE_TIBET_Z);
#endif
  printf("  %s <filename>         output CSW filename\n",
         QB_CLI_S_OPFILE_CSW);
  printf("                        (%s"
#ifdef QB_HAVE_ZLIB
         ", %s"
#endif
         " and %s may be provided in combination)\n",
         QB_CLI_S_OPFILE_TIBET,
#ifdef QB_HAVE_ZLIB
         QB_CLI_S_OPFILE_TIBET_Z,
#endif
         QB_CLI_S_OPFILE_CSW);
  printf("  %s    do NOT autocorrect CSW phase for MakeUEF\n",
         QB_CLI_S_NO_CSW_POLARITY_CORRECTION);

  printf("\nInput options:\n");
  printf("  %s (L|R)              pick channel from stereo file (left is default)\n",
         QB_CLI_S_CHANNEL);

  printf("  %s                    bandpass-filter the input (not usually helpful)\n",
         QB_CLI_S_WALK_DO_FILTERING);
  printf("  %s                    normalise the input (not usually helpful)\n",
         QB_CLI_S_NORMALISE);

  // 2.0.3: default is -s sync, was -s pll in 2.0.2; changed order to list PLL second; added reliability hints
  printf("%sSync options:\n", spacer);
  printf("  %s <method>           sync recovery: <method> may be one of:\n",
         QB_CLI_S_SYNC_METHOD);
  printf("                          %s     frequency domain (fast, reliable) [default];\n",
         QB_CLI_S_SYNC_METHOD_GOERTZEL);
  printf("                          %s      PLL              (slow, reliable);\n",
         QB_CLI_S_SYNC_METHOD_PLL);
  printf("                          %s     time domain      (slow, unreliable).\n",
         QB_CLI_S_SYNC_METHOD_WALK);

  printf("\nPhase detection options:\n");
  printf("  %s <pmode>            input phase mode; <pmode> may be one of:\n",
         QB_CLI_S_PHASE_MODE);
  printf("                          %s         fixed phase, autodetected [default]\n",
         QB_CLI_S_PHASE_SCHEME_FIXED_AUTO);
  printf("                          %s        per-block phase, autodetected\n",
         QB_CLI_S_PHASE_SCHEME_BLOCK);
  printf("                          %s   or '%s'   force fixed 0-degree phase\n",
         QB_CLI_S_PHASE_SCHEME_FIXED_0,
         QB_CLI_S_PHASE_SCHEME_FIXED_ALT_0);
  printf("                          %s  or '%s'   force fixed 90-degree phase\n",
         QB_CLI_S_PHASE_SCHEME_FIXED_90,
         QB_CLI_S_PHASE_SCHEME_FIXED_ALT_90);
  printf("                          %s or '%s'   force fixed 180-degree phase\n",
         QB_CLI_S_PHASE_SCHEME_FIXED_180,
         QB_CLI_S_PHASE_SCHEME_FIXED_ALT_180);
  printf("                          %s or '%s'   force fixed 270-degree phase\n",
         QB_CLI_S_PHASE_SCHEME_FIXED_270,
         QB_CLI_S_PHASE_SCHEME_FIXED_ALT_270);
  printf("  %s <m>       force phase detection scheme for %s %s/%s\n"
         "                        (the default is to match the sync method);\n"
         "                        <m> may be either:\n",
         QB_CLI_S_PHASE_DETECTION_METHOD,
         QB_CLI_S_PHASE_MODE,
         QB_CLI_S_PHASE_SCHEME_FIXED_AUTO,
         QB_CLI_S_PHASE_SCHEME_BLOCK);
  printf("                          %s     use PLL phase detection method [default]\n",
         QB_CLI_S_PHASE_DETECT_PLL);
  printf("                          %s    use walk phase detection method\n",
         QB_CLI_S_PHASE_DETECT_WALK);

  printf("%sScanning options (no output file is produced):\n", spacer);
  printf("  %s    search best value of %s (slow!)\n", QB_CLI_S_FIND_BEST_2400_POWER, QB_CLI_S_SCALE_2400_POWER);
        
  printf("%sGeneral processing options:\n", spacer);
  printf("  %s  <x.x> 2400 Hz power scaling factor (%.1f -> %.1f)\n",
         QB_CLI_S_SCALE_2400_POWER, QB_SCALE_2400_POWER_MIN, QB_SCALE_2400_POWER_MAX);
  //printf("  %s     skip optimal shift search (cannot use with %s %s)\n",
  //       QB_CLI_S_NO_SHIFT_SEARCH, QB_CLI_S_SYNC_METHOD, QB_CLI_S_SYNC_METHOD_GOERTZEL);
  printf("  %s   <s>   force shift in smps (%d -> %d, cannot use with %s %s)\n",
         QB_CLI_S_FORCE_SHIFT,
         -QB_SPAN_SHIFT_SEARCH_HALF_RANGE,
         QB_SPAN_SHIFT_SEARCH_HALF_RANGE,
         QB_CLI_S_SYNC_METHOD, QB_CLI_S_SYNC_METHOD_GOERTZEL);
#ifdef QB_SCHMITT_TRIGGER_THRESH_OPTIONS
  printf("  %s <x.x> set power ratio threshold for silence detection\n"
         "                          (%.3f -> %f, default %f)\n",
         QB_CLI_S_SILENT_THRESH, QB_SILENT_THRESH_MIN, QB_SILENT_THRESH_MAX, QB_SILENT_THRESH_DEFAULT);
  printf("  %s <x.x> set power ratio threshold for signal detection\n"
         "                          (%f -> %f, default %f)\n",
         QB_CLI_S_SIGNAL_THRESH, QB_SIGNAL_THRESH_MIN, QB_SIGNAL_THRESH_MAX, QB_SIGNAL_THRESH_DEFAULT);
#endif // QB_SCHMITT_TRIGGER_THRESH_OPTIONS
  printf("  %s   <n>   set threshold for silencing weak signal spans\n"
         "                          (0-100, default %d, 0 to disable)\n",
         QB_CLI_S_HUSH_THRESH, (int) round(QB_HUSH_DEFAULT_PERCENT));

  //printf("  %s <x.x> set summed power ratio threshold for post-converting\n"
  //       "                        signal spans to silent\n"
  //       "                          (%f -> %f, default %f)\n",
  //       QB_CLI_S_SUMMED_THRESH, QB_SUMMED_THRESH_MIN, QB_SUMMED_THRESH_MAX, QB_SUMMED_THRESH_DEFAULT);
  
  printf("%sError-correction options:\n", spacer);
  printf("  %s  don't fix odd runs by insert (PLL) or constraint (freq)\n                          (not with %s %s)\n",
         QB_CLI_S_NO_DEFAULT_ODD_FIX, QB_CLI_S_SYNC_METHOD, QB_CLI_S_SYNC_METHOD_WALK);
  printf("  %s <policy>           fix adjacent odd-run pairs by atom flipping\n"
         "                        <policy> may be one of the following: \n",
         QB_CLI_S_ERROR_CORRECTION);
  printf("                          %s, %s, %s, %s, %s, %s.\n",
         QB_CLI_S_ERRCORR_SINGLETON_WINS,
         QB_CLI_S_ERRCORR_CONFIDENCE_WINS,
         QB_CLI_S_ERRCORR_ZERO_WINS,
         QB_CLI_S_ERRCORR_ONE_WINS,
         QB_CLI_S_ERRCORR_EARLY_WINS,
         QB_CLI_S_ERRCORR_LATE_WINS);
  
  printf("%sDiagnostic options:\n", spacer);
  printf("  %s <dir>   save intermediate data as WAV files to directory\n",
         QB_CLI_S_DBG_INSPECT_DIR);

  printf("%sOther options:\n", spacer);
  printf("  %s                    quieter; suppress progress updates\n",
         QB_CLI_S_QUIET);
  printf("  %s                    verbose (extra warnings)\n",
         QB_CLI_S_VERBOSE);
  printf("  %s                    show this help text\n",
         QB_CLI_S_HELP);

}

static void print_dupe_cli_error (char *swch) {
  fprintf(QB_ERR, "E: Command-line switch %s may not be specified twice.\n", swch);
}

#define QB_CLI_STATE_OPTIONS                1
#define QB_CLI_STATE_OPFILE_CSW             2
#define QB_CLI_STATE_OPFILE_TIBET           3
#define QB_CLI_STATE_OPFILE_TIBET_Z         4
#define QB_CLI_STATE_IPFILE                 5
#define QB_CLI_STATE_SYNC_METHOD            6
#define QB_CLI_STATE_CHANNEL                7
#define QB_CLI_STATE_PHASE_MODE             8
#define QB_CLI_STATE_INSPECT_FILES_DIR      9
#define QB_CLI_STATE_ERR_CORRECT            10
#define QB_CLI_STATE_PHASE_DETECTION_METHOD 11
#define QB_CLI_STATE_PLL_FORCED_DELAY       12
#define QB_CLI_STATE_SCALE_2400_POWER       13
#define QB_CLI_STATE_SILENT_THRESH          14
#define QB_CLI_STATE_SIGNAL_THRESH          15
#define QB_CLI_STATE_SUMMED_THRESH          16


static qb_err_t qb_cli (qb_t *qb, int argc, char *argv[]) {

  int a;
  u8_t have_quiet;
  u8_t have_sync_method;
  u8_t have_do_filtering;
  u8_t have_phase_scheme;
  u8_t have_channel;
  u8_t have_inspect_files_dir;
  u8_t have_atomflip_correction;
  u8_t have_phase_detection_method;
  u8_t have_forced_shift;
  u8_t have_csw_file;
  u8_t have_scale_2400_power;
  u8_t have_verbose;
  u8_t have_normalise;
  u8_t have_tibet_file;
  u8_t have_tibet_z_file;
  u8_t have_no_csw_polarity_correction;
  u8_t have_no_generic_odd_run_fix;
#ifdef QB_SCHMITT_TRIGGER_THRESH_OPTIONS
  u8_t have_silent_thresh;
  u8_t have_signal_thresh;
#endif
  u8_t have_hush_thresh;
  //u8_t have_scan_hf_pwr; // 2.0.4
  
  u8_t need_phase_detection;

  qb_err_t e;
  u8_t state;
  size_t len;
  
  char *endpos_for_strtol;
  long ln;
  
  have_forced_shift                   = 0;
  have_quiet                          = 0;
  have_sync_method                    = 0;
  have_do_filtering                   = 0;
  have_phase_scheme                   = 0;
  have_channel                        = 0;
  have_inspect_files_dir              = 0;
  have_atomflip_correction            = 0;
  have_phase_detection_method         = 0;
  have_csw_file                       = 0;
  have_scale_2400_power               = 0;
  have_verbose                        = 0;
  have_normalise                      = 0;
  have_tibet_file                     = 0;
  have_tibet_z_file                   = 0;
  have_no_csw_polarity_correction     = 0;
  have_no_generic_odd_run_fix         = 0;
#ifdef QB_SCHMITT_TRIGGER_THRESH_OPTIONS
  have_silent_thresh                  = 0;
  have_signal_thresh                  = 0;
#endif
  have_hush_thresh                    = 0;
    
  state = QB_CLI_STATE_OPTIONS;
  e = QB_E_OK;
  
  if (argc < 2) {
    qb_usage(argv[0]);
    return QB_E_CLI;
  }
  
  // default to PLL sync
  qb->sync_method            = QB_SYNC_METHOD_PLL;
  // default to 1.0 2400-Hz scaler, UNLESS -s freq, in which case use 0.9
  qb->scaler_for_2400_power  = 1.0;
  
  qb->silence_on_frac        = QB_SILENT_THRESH_DEFAULT;
  qb->signal_on_frac         = QB_SIGNAL_THRESH_DEFAULT;
  qb->summed_silence_on_frac = QB_HUSH_THRESH_DEFAULT;
  
  need_phase_detection = 0;
  
  for (a=1; a < argc; a++) {
  
    double d;
    s64_t i;
    
    len = strlen(argv[a]);
    
    if (0 == len) {
      fprintf(QB_ERR, "E: Empty command line argument.\n");
      e = QB_E_CLI;
      break;
    }
    
    if (QB_CLI_STATE_OPFILE_CSW == state) {
      qb->csw_opfile = qb_malloc(len+1);
      if (NULL == qb->csw_opfile) {
        fprintf(QB_ERR, "E: Out of memory allocating CSW filename copy.\n");
        e = QB_E_MALLOC;
        break;
      }
      memcpy(qb->csw_opfile, argv[a], len+1);
      state = QB_CLI_STATE_OPTIONS;
    } else if (QB_CLI_STATE_OPFILE_TIBET == state) {
      qb->tibet_opfile = qb_malloc(len+1);
      if (NULL == qb->tibet_opfile) {
        fprintf(QB_ERR, "E: Out of memory allocating TIBET filename copy.\n");
        e = QB_E_MALLOC;
        break;
      }
      memcpy(qb->tibet_opfile, argv[a], len+1);
      state = QB_CLI_STATE_OPTIONS;
    } else if (QB_CLI_STATE_OPFILE_TIBET_Z == state) {
      qb->tibet_z_opfile = qb_malloc(len+1);
      if (NULL == qb->tibet_z_opfile) {
        fprintf(QB_ERR, "E: Out of memory allocating TIBETZ filename copy.\n");
        e = QB_E_MALLOC;
        break;
      }
      memcpy(qb->tibet_z_opfile, argv[a], len+1);
      state = QB_CLI_STATE_OPTIONS;
    } else if (QB_CLI_STATE_OPTIONS == state) {
      if ( argv[a][0] == '-' ) {
        if ( 0 == strcmp ( argv[a], QB_CLI_S_OPFILE_CSW ) ) {
          if (have_csw_file) {
            print_dupe_cli_error(QB_CLI_S_OPFILE_CSW);
            e = QB_E_CLI;
            break;
          }
          state = QB_CLI_STATE_OPFILE_CSW;
          have_csw_file = 1;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_OPFILE_TIBET ) ) {
          if (have_tibet_file) {
            print_dupe_cli_error(QB_CLI_S_OPFILE_TIBET);
            e = QB_E_CLI;
            break;
          }
          state = QB_CLI_STATE_OPFILE_TIBET;
          have_tibet_file = 1;
#ifdef QB_HAVE_ZLIB
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_OPFILE_TIBET_Z ) ) {
          if (have_tibet_z_file) {
            print_dupe_cli_error(QB_CLI_S_OPFILE_TIBET_Z);
            e = QB_E_CLI;
            break;
          }
          state = QB_CLI_STATE_OPFILE_TIBET_Z;
          have_tibet_z_file = 1;
#endif
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_OVERWRITE ) ) {
          if (qb->allow_overwrite) {
            print_dupe_cli_error(QB_CLI_S_OVERWRITE);
            e = QB_E_CLI;
            break;
          }
          qb->allow_overwrite = 1;
          qb->inspect.allow_overwrite = 1;
          //~ have_overwrite = 1;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_QUIET ) ) {
          if (have_quiet) {
            print_dupe_cli_error(QB_CLI_S_QUIET);
            e = QB_E_CLI;
            break;
          }
          have_quiet = 1;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_VERBOSE ) ) {
          if (have_verbose) {
            print_dupe_cli_error(QB_CLI_S_VERBOSE);
            e = QB_E_CLI;
            break;
          }
          have_verbose = 1;
          qb->verbose = 1;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_NORMALISE ) ) {
          if (have_normalise) {
            print_dupe_cli_error(QB_CLI_S_NORMALISE);
            e = QB_E_CLI;
            break;
          }
          have_normalise = 1;
          qb->normalise = 1;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_SYNC_METHOD ) ) {
          if (have_sync_method) {
            print_dupe_cli_error (QB_CLI_S_SYNC_METHOD);
            e = QB_E_CLI;
            break;
          }
          state = QB_CLI_STATE_SYNC_METHOD;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_DBG_INSPECT_DIR ) ) {
          if (have_inspect_files_dir) {
            print_dupe_cli_error (QB_CLI_S_DBG_INSPECT_DIR);
            e = QB_E_CLI;
            break;
          }
          have_inspect_files_dir = 1;
          state = QB_CLI_STATE_INSPECT_FILES_DIR;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_HELP ) ) {
          qb_usage(argv[0]);
          e = QB_E_CLI;
          break;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_CHANNEL ) ) {
          if (have_channel) {
            print_dupe_cli_error (QB_CLI_S_CHANNEL);
            e = QB_E_CLI;
            break;
          }
          have_channel = 1;
          state = QB_CLI_STATE_CHANNEL;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_PHASE_MODE ) ) {
          if (have_phase_scheme) {
            print_dupe_cli_error (QB_CLI_S_PHASE_MODE);
            e = QB_E_CLI;
            break;
          }
          have_phase_scheme = 1;
          state = QB_CLI_STATE_PHASE_MODE;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_WALK_DO_FILTERING ) ) {
          if (have_do_filtering) {
            print_dupe_cli_error (QB_CLI_S_WALK_DO_FILTERING);
            e = QB_E_CLI;
            break;
          }
          have_do_filtering = 1;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_ERROR_CORRECTION ) ) {
          if (have_atomflip_correction) {
            print_dupe_cli_error (QB_CLI_S_ERROR_CORRECTION);
            e = QB_E_CLI;
            break;
          }
          have_atomflip_correction = 1;
          state = QB_CLI_STATE_ERR_CORRECT;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_PHASE_DETECTION_METHOD ) ) {
          if (have_phase_detection_method) {
            print_dupe_cli_error(QB_CLI_S_PHASE_DETECTION_METHOD);
            e = QB_E_CLI;
            break;
          }
          state = QB_CLI_STATE_PHASE_DETECTION_METHOD;
          have_phase_detection_method = 1;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_FORCE_SHIFT ) ) {
          if (have_forced_shift) {
            print_dupe_cli_error(QB_CLI_S_FORCE_SHIFT);
            e = QB_E_CLI;
            break;
          }
          have_forced_shift = 1;
          state = QB_CLI_STATE_PLL_FORCED_DELAY;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_SCALE_2400_POWER ) ) {
          if (have_scale_2400_power) {
            print_dupe_cli_error(QB_CLI_S_SCALE_2400_POWER);
            e = QB_E_CLI;
            break;
          }
          state = QB_CLI_STATE_SCALE_2400_POWER;
          have_scale_2400_power = 1;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_NO_CSW_POLARITY_CORRECTION ) ) {
          if (have_no_csw_polarity_correction) {
            print_dupe_cli_error(QB_CLI_S_NO_CSW_POLARITY_CORRECTION);
            e = QB_E_CLI;
            break;
          }
          have_no_csw_polarity_correction = 1;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_NO_DEFAULT_ODD_FIX ) ) {
          if (have_no_generic_odd_run_fix) {
            print_dupe_cli_error(QB_CLI_S_NO_DEFAULT_ODD_FIX);
            e = QB_E_CLI;
            break;
          }
          have_no_generic_odd_run_fix = 1;
#ifdef QB_SCHMITT_TRIGGER_THRESH_OPTIONS
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_SILENT_THRESH ) ) {
          if (have_silent_thresh) {
            print_dupe_cli_error(QB_CLI_S_SILENT_THRESH);
            e = QB_E_CLI;
            break;
          }
          state = QB_CLI_STATE_SILENT_THRESH;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_SIGNAL_THRESH ) ) {
          if (have_signal_thresh) {
            print_dupe_cli_error(QB_CLI_S_SIGNAL_THRESH);
            e = QB_E_CLI;
            break;
          }
          state = QB_CLI_STATE_SIGNAL_THRESH;
#endif // QB_SCHMITT_TRIGGER_THRESH_OPTIONS
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_HUSH_THRESH ) ) {
          if (have_hush_thresh) {
            print_dupe_cli_error(QB_CLI_S_HUSH_THRESH);
            e = QB_E_CLI;
            break;
          }
          state = QB_CLI_STATE_SUMMED_THRESH;
          have_hush_thresh = 1;
        } else if ( 0 == strcmp ( argv[a], QB_CLI_S_FIND_BEST_2400_POWER) ) { // 2.0.4
          if (qb->find_best_hf_pwr) {
            print_dupe_cli_error(QB_CLI_S_FIND_BEST_2400_POWER);
            e = QB_E_CLI;
            break;
          }
          qb->find_best_hf_pwr = 1;
        } else if ( (argv[a][1] == '-') && (len == 2) ) { // end of options; CAUTION: this one must be at the end!! 
          state = QB_CLI_STATE_IPFILE;
        } else {
          fprintf(QB_ERR, "E: Unrecognised option \"%s\".\n", argv[a]); 
          e = QB_E_CLI;
          break;
        }
      } else { // didn't start with "-" so it must be the first input file
        state = QB_CLI_STATE_IPFILE;
        a--; // do this one again
      }
#ifdef QB_SCHMITT_TRIGGER_THRESH_OPTIONS
    } else if (QB_CLI_STATE_SILENT_THRESH == state) {
      // parse float
      d = 0.0;
      if ( QB_E_OK != cli_parse_decimal (argv[a], 0, &d) ) {
        fprintf(QB_ERR, "E: \"%s\" was not a legal decimal number.\n", argv[a]);
        e = QB_E_CLI;
        break;
      } else if (d < QB_SILENT_THRESH_MIN) {
        fprintf(QB_ERR, "E: %s value \"%s\" was too small (min. %lf).\n",
                QB_CLI_S_SILENT_THRESH, argv[a], QB_SILENT_THRESH_MIN);
        e = QB_E_CLI;
        break;
      } else if (d > QB_SILENT_THRESH_MAX) {
        fprintf(QB_ERR, "E: %s value \"%s\" was too large (max. %lf).\n",
                QB_CLI_S_SILENT_THRESH, argv[a], QB_SILENT_THRESH_MAX);
        e = QB_E_CLI;
        break;
      }
      qb->silence_on_frac = d;
      state = QB_CLI_STATE_OPTIONS;
    } else if (QB_CLI_STATE_SIGNAL_THRESH == state) {
      // parse float
      d = 0.0;
      if ( QB_E_OK != cli_parse_decimal (argv[a], 0, &d) ) {
        fprintf(QB_ERR, "E: \"%s\" was not a legal decimal number.\n", argv[a]);
        e = QB_E_CLI;
        break;
      } else if (d < QB_SIGNAL_THRESH_MIN) {
        fprintf(QB_ERR, "E: %s value \"%s\" was too small (min. %lf).\n",
                QB_CLI_S_SIGNAL_THRESH, argv[a], QB_SIGNAL_THRESH_MIN);
        e = QB_E_CLI;
        break;
      } else if (d > QB_SIGNAL_THRESH_MAX) {
        fprintf(QB_ERR, "E: %s value \"%s\" was too large (max. %lf).\n",
                QB_CLI_S_SIGNAL_THRESH, argv[a], QB_SIGNAL_THRESH_MAX);
        e = QB_E_CLI;
        break;
      }
      qb->signal_on_frac = d;
      state = QB_CLI_STATE_OPTIONS;
#endif // QB_SCHMITT_TRIGGER_THRESH_OPTIONS
    } else if (QB_CLI_STATE_SUMMED_THRESH == state) {
      // parse int
      i = 0;
      if ( QB_E_OK != cli_parse_int (argv[a], 0, 3, &i) ) {
        fprintf(QB_ERR, "E: \"%s\" was not a legal value (expect 0-100).\n", argv[a]);
        e = QB_E_CLI;
        break;
      } else if (i > 100) {
        fprintf(QB_ERR, "E: %s value \"%s\" was too large (max. 100).\n",
                QB_CLI_S_HUSH_THRESH, argv[a]);
        e = QB_E_CLI;
        break;
      }
      // work stuff out
      qb->summed_silence_on_frac = ((((float) i) / 100.0f) * QB_HUSH_RANGE) + QB_HUSH_THRESH_MIN;
      //qb->summed_silence_on_frac = d;
      state = QB_CLI_STATE_OPTIONS;
    } else if (QB_CLI_STATE_SCALE_2400_POWER == state) {
      // parse float
      d = 0.0;
      if ( QB_E_OK != cli_parse_decimal (argv[a], 0, &d) ) {
        fprintf(QB_ERR, "E: \"%s\" was not a legal decimal number.\n", argv[a]);
        e = QB_E_CLI;
        break;
      } else if (d < QB_SCALE_2400_POWER_MIN) {
        fprintf(QB_ERR, "E: %s value \"%s\" was too small (min. %.1lf).\n",
                QB_CLI_S_SCALE_2400_POWER, argv[a], QB_SCALE_2400_POWER_MIN);
        e = QB_E_CLI;
        break;
      } else if (d > QB_SCALE_2400_POWER_MAX) {
        fprintf(QB_ERR, "E: %s value \"%s\" was too large (max. %.1lf).\n",
                QB_CLI_S_SCALE_2400_POWER, argv[a], QB_SCALE_2400_POWER_MAX);
        e = QB_E_CLI;
        break;
      }
      qb->scaler_for_2400_power = (float) d;
      qb->forced_scaler_for_2400_power = 1; // only used by CLI summary printout
      state = QB_CLI_STATE_OPTIONS;
    } else if (QB_CLI_STATE_INSPECT_FILES_DIR == state) {
      len = strlen(argv[a]);
      qb->inspect.dir = qb_malloc (len + 1);
      if (NULL == qb->inspect.dir) {
        fprintf(QB_ERR, "E: Out of memory allocating inspect dirname copy.\n");
        e = QB_E_MALLOC;
        break;
      }
      memcpy (qb->inspect.dir, argv[a], len+1);
      state = QB_CLI_STATE_OPTIONS;
    } else if (QB_CLI_STATE_IPFILE == state) {
      qb->ipfile_path = qb_malloc(len+1);
      if (NULL == qb->ipfile_path) {
        fprintf(QB_ERR, "E: Out of memory allocating input filename copy.\n");
        e = QB_E_MALLOC;
        break;
      }
      memcpy(qb->ipfile_path, argv[a], len+1);
      // no further arguments are allowed after the input file
      if (a != (argc-1)) {
        fprintf(QB_ERR, "E: Junk arguments found at end of command-line.\n");
        e = QB_E_CLI;
        break;
      }
    } else if (QB_CLI_STATE_SYNC_METHOD == state) {
      if ( 0 == strcmp (argv[a], QB_CLI_S_SYNC_METHOD_WALK) ) {
        qb->sync_method = QB_SYNC_METHOD_WALK;
      } else if ( 0 == strcmp (argv[a], QB_CLI_S_SYNC_METHOD_GOERTZEL ) ) {
        qb->sync_method = QB_SYNC_METHOD_GOERTZEL;
      } else if ( 0 == strcmp (argv[a], QB_CLI_S_SYNC_METHOD_PLL ) ) {
        qb->sync_method = QB_SYNC_METHOD_PLL;
      } else {
        fprintf(QB_ERR, "E: Unrecognised sync method \"%s\".\n", argv[a]);
        fprintf(QB_ERR, "   Legal options are \"%s\", \"%s\" or \"%s\".\n",
                QB_CLI_S_SYNC_METHOD_PLL,
                QB_CLI_S_SYNC_METHOD_WALK,
                QB_CLI_S_SYNC_METHOD_GOERTZEL);
        e = QB_E_CLI;
        break;
      } 
      have_sync_method = 1;
      state = QB_CLI_STATE_OPTIONS;
    } else if (QB_CLI_STATE_CHANNEL == state) {
      e = parse_source_selector_simple (argv[a], qb);
      state = QB_CLI_STATE_OPTIONS;
    } else if (QB_CLI_STATE_PHASE_MODE == state) {
      if ( 0 == strcmp (argv[a], QB_CLI_S_PHASE_SCHEME_FIXED_AUTO) ) {
        qb->phase_scheme = QB_PHASE_SCHEME_FIXED_AUTO;
      } else if (    ( 0 == strcmp (argv[a], QB_CLI_S_PHASE_SCHEME_FIXED_0 ) )
                  || ( 0 == strcmp (argv[a], QB_CLI_S_PHASE_SCHEME_FIXED_ALT_0 ) ) ) {
        qb->phase_scheme = QB_PHASE_SCHEME_FIXED_0;
      } else if (    ( 0 == strcmp (argv[a], QB_CLI_S_PHASE_SCHEME_FIXED_90 ) )
                  || ( 0 == strcmp (argv[a], QB_CLI_S_PHASE_SCHEME_FIXED_ALT_90 ) ) ) {
        qb->phase_scheme = QB_PHASE_SCHEME_FIXED_90;
      } else if (    ( 0 == strcmp (argv[a], QB_CLI_S_PHASE_SCHEME_FIXED_180 ) )
                  || ( 0 == strcmp (argv[a], QB_CLI_S_PHASE_SCHEME_FIXED_ALT_180 ) ) ) {
        qb->phase_scheme = QB_PHASE_SCHEME_FIXED_180;
      } else if (    ( 0 == strcmp (argv[a], QB_CLI_S_PHASE_SCHEME_FIXED_270 ) )
                  || ( 0 == strcmp (argv[a], QB_CLI_S_PHASE_SCHEME_FIXED_ALT_270 ) ) ) {
        qb->phase_scheme = QB_PHASE_SCHEME_FIXED_270;
      } else if ( 0 == strcmp (argv[a], QB_CLI_S_PHASE_SCHEME_BLOCK ) ) {
        qb->phase_scheme = QB_PHASE_SCHEME_BLOCK_AUTO;
      } else {
        fprintf(QB_ERR, "E: Unrecognised phase scheme \"%s\".\n", argv[a]);
        e = QB_E_CLI;
        break;
      }
      state = QB_CLI_STATE_OPTIONS;
    } else if (QB_CLI_STATE_ERR_CORRECT == state) {
    
      if ( 0 == strcmp (QB_CLI_S_ERRCORR_SINGLETON_WINS, argv[a]) ) {
        qb->err_correction_strategy = QB_ERRCORR_SINGLETON_WINS;
      } else if ( 0 == strcmp (QB_CLI_S_ERRCORR_CONFIDENCE_WINS, argv[a]) ) {
        qb->err_correction_strategy = QB_ERRCORR_CONFIDENCE_WINS;
      } else if ( 0 == strcmp (QB_CLI_S_ERRCORR_ZERO_WINS, argv[a]) ) {
        qb->err_correction_strategy = QB_ERRCORR_ZERO_WINS;
      } else if ( 0 == strcmp (QB_CLI_S_ERRCORR_ONE_WINS, argv[a]) ) {
        qb->err_correction_strategy = QB_ERRCORR_ONE_WINS;
      } else if ( 0 == strcmp (QB_CLI_S_ERRCORR_EARLY_WINS, argv[a]) ) {
        qb->err_correction_strategy = QB_ERRCORR_EARLY_WINS;
      } else if ( 0 == strcmp (QB_CLI_S_ERRCORR_LATE_WINS, argv[a]) ) {
        qb->err_correction_strategy = QB_ERRCORR_LATE_WINS;
      } else {
      
        fprintf(QB_ERR, "E: Unrecognised error correction mode \"%s\".\n", argv[a]);
        fprintf(QB_ERR, "E: Legal options are %s, %s, %s, %s, %s, %s.\n",
                QB_CLI_S_ERRCORR_SINGLETON_WINS,
                QB_CLI_S_ERRCORR_CONFIDENCE_WINS,
                QB_CLI_S_ERRCORR_ZERO_WINS,
                QB_CLI_S_ERRCORR_ONE_WINS,
                QB_CLI_S_ERRCORR_EARLY_WINS,
                QB_CLI_S_ERRCORR_LATE_WINS);
        e = QB_E_CLI;
        break;
      }
      state = QB_CLI_STATE_OPTIONS;
    } else if (QB_CLI_STATE_PHASE_DETECTION_METHOD == state) {
      if ( 0 == strcmp (QB_CLI_S_PHASE_DETECT_PLL, argv[a]) ) {
        qb->phase_detect_method = QB_PHASE_DETECT_METHOD_PLL;
      } else if ( 0 == strcmp (QB_CLI_S_PHASE_DETECT_WALK, argv[a]) ) {
        qb->phase_detect_method = QB_PHASE_DETECT_METHOD_WALK;
      } else {
        fprintf(QB_ERR, "E: Unrecognised phase detection method \"%s\".\n", argv[a]);
        fprintf(QB_ERR, "E: Legal options are \"%s\" or \"%s\".\n",
                QB_CLI_S_PHASE_DETECT_PLL, QB_CLI_S_PHASE_DETECT_WALK);
        e = QB_E_CLI;
        break;
      }
      state = QB_CLI_STATE_OPTIONS;
    } else if (QB_CLI_STATE_PLL_FORCED_DELAY == state) {
      ln = strtol (argv[a], &endpos_for_strtol, 10);
      if (endpos_for_strtol != (argv[a] + len) ) {
        fprintf(QB_ERR, "PLL carrier-offset value was not an integer: %s\n", argv[a]);
        e = QB_E_CLI;
        break;
      }
      if ( (ln < -QB_SPAN_SHIFT_SEARCH_HALF_RANGE ) || (ln > QB_SPAN_SHIFT_SEARCH_HALF_RANGE) ) {
        fprintf(QB_ERR, "Forced shift value (%s) was illegal (%ld; min. %d, max. %d).\n",
                QB_CLI_S_FORCE_SHIFT,
                ln,
                -QB_SPAN_SHIFT_SEARCH_HALF_RANGE,
                QB_SPAN_SHIFT_SEARCH_HALF_RANGE);
        e = QB_E_CLI;
        break;
      }
      qb->forced_shift    = 0xff & ln;
      //qb->no_shift_search = 1;
      state = QB_CLI_STATE_OPTIONS;
    } else {
      fprintf(QB_ERR, "B: %s: weirdo state %u.\n", QB_FUNC_M, state);
      e = QB_E_BUG;
      break;
    }
  } // next argv

  if (QB_E_OK == e) {
  
    do { // try {
    
      if (NULL == qb->ipfile_path) {
        fprintf(QB_ERR, "E: No input files were provided!\n");
        e = QB_E_CLI;
        break;
      }
      
      if ( ! have_sync_method ) {
        // 2.0.3: was -s pll, now -s freq by default
        qb->sync_method = QB_SYNC_METHOD_GOERTZEL;
      }
      
      // ban phase stuff with -s freq (since -s freq is phase-agnostic)
      if (QB_SYNC_METHOD_GOERTZEL == qb->sync_method) {
      
        // -s freq with --phase-det
        if (have_phase_detection_method) {
          fprintf(QB_ERR, "E: Cannot use %s %s with %s.\n",
                  QB_CLI_S_SYNC_METHOD,
                  QB_CLI_S_SYNC_METHOD_GOERTZEL,
                  QB_CLI_S_PHASE_DETECTION_METHOD);
          e = QB_E_CLI;
          break;
        }
      
        // -s freq with -p
        if (have_phase_scheme) {
          fprintf(QB_ERR, "E: Cannot use %s with %s %s.\n",
                  QB_CLI_S_PHASE_MODE,
                  QB_CLI_S_SYNC_METHOD,
                  QB_CLI_S_SYNC_METHOD_GOERTZEL);
          e = QB_E_CLI;
          break;
        }
        
      }
    
      
      if (    have_phase_detection_method
           && have_phase_scheme
           && (    (QB_PHASE_SCHEME_FIXED_0   == qb->phase_scheme )
                || (QB_PHASE_SCHEME_FIXED_90  == qb->phase_scheme )
                || (QB_PHASE_SCHEME_FIXED_180 == qb->phase_scheme )
                || (QB_PHASE_SCHEME_FIXED_270 == qb->phase_scheme ) ) ) {
        fprintf(QB_ERR,
                "E: %s requires %s %s or %s %s.\n",
                QB_CLI_S_PHASE_DETECTION_METHOD,
                QB_CLI_S_PHASE_MODE,
                QB_CLI_S_PHASE_SCHEME_BLOCK,
                QB_CLI_S_PHASE_MODE,
                QB_CLI_S_PHASE_SCHEME_FIXED_AUTO);
        e = QB_E_CLI;
        break;
      }

      // 2.0.4
      if (qb->find_best_hf_pwr) {
        // ban output options with scan modes
        if (have_tibet_file || have_tibet_z_file || have_csw_file || qb->allow_overwrite) {
          fprintf(QB_ERR,
                  "E: Output options (%s, %s, %s, %s) not allowed with %s.\n",
                  QB_CLI_S_OPFILE_CSW,
                  QB_CLI_S_OPFILE_TIBET,
                  QB_CLI_S_OPFILE_TIBET_Z,
                  QB_CLI_S_OVERWRITE,
                  QB_CLI_S_FIND_BEST_2400_POWER);
          e = QB_E_CLI;
          break;
        }
        if (have_inspect_files_dir) {
          fprintf(QB_ERR,
            "E: %s not allowed with %s.\n",
            QB_CLI_S_DBG_INSPECT_DIR,
            QB_CLI_S_FIND_BEST_2400_POWER);
          e = QB_E_CLI;
          break;
        }
      }

      if (have_do_filtering) {
        qb->do_filtering = 1;
      }
      
      if (have_quiet) {
        qb->quieter = 1;
      }
      
      if (have_inspect_files_dir) {
        qb->inspect.enabled = 1;
      }
      
      if ( ! have_tibet_file && ! have_tibet_z_file && ! have_csw_file && /* 2.0.4 */ ! qb->find_best_hf_pwr) {
        // if no filenames provided at all (and not scan mode), just assume default TIBET filename
        len = strlen(QB_OPFILE_DEFAULT);
        qb->tibet_opfile = qb_malloc(len+1);
        if (NULL == qb->tibet_opfile) {
          fprintf(QB_ERR, "E: Out of memory allocating default TIBET output file name.\n");
          e = QB_E_MALLOC;
          break;
        }
        memcpy(qb->tibet_opfile, QB_OPFILE_DEFAULT, len+1);
        qb->using_default_op_file = 1;
        have_tibet_file = 1;
      }
      
      if (have_no_csw_polarity_correction) {
        if ( ! have_csw_file ) {
          fprintf(QB_ERR, "E: %s requires CSW file output.\n",
                  QB_CLI_S_NO_CSW_POLARITY_CORRECTION);
          e = QB_E_CLI;
          break;
        }
        qb->no_csw_polarity_correction = 1;
      }
      
      need_phase_detection =    (QB_SYNC_METHOD_PLL  == qb->sync_method)
                             || (QB_SYNC_METHOD_WALK == qb->sync_method);
      
      // need phase detection (walk, pll) but have no phase scheme
      // default to fixed, auto
      if ( need_phase_detection && (0 == qb->phase_scheme) ) {
        qb->phase_scheme = QB_PHASE_SCHEME_FIXED_AUTO;
      }
      
      // for walk and pll modes, we need a phase detection method
      // (freq. sync mode doesn't care about phase)
      // pick the one by default that matches -s
      if ( need_phase_detection && ! have_phase_detection_method ) {
        if (QB_SYNC_METHOD_PLL == qb->sync_method) {
          qb->phase_detect_method = QB_PHASE_DETECT_METHOD_PLL;
        } else {
          qb->phase_detect_method = QB_PHASE_DETECT_METHOD_WALK;
        }
      }
      
      if (have_no_generic_odd_run_fix) {
        if (QB_SYNC_METHOD_WALK == qb->sync_method) {
          fprintf(QB_ERR, "E: Cannot use %s with %s %s.\n",
                  QB_CLI_S_NO_DEFAULT_ODD_FIX,
                  QB_CLI_S_SYNC_METHOD,
                  QB_CLI_S_SYNC_METHOD_WALK);
          e = QB_E_CLI;
          break;
        }
        qb->no_generic_odd_run_fix = 1;
      }
      
      if ( have_forced_shift ) {
        if (QB_SYNC_METHOD_GOERTZEL == qb->sync_method) {
          fprintf(QB_ERR, "E: Cannot use %s with %s %s.\n",
                  QB_CLI_S_FORCE_SHIFT,
                  QB_CLI_S_SYNC_METHOD,
                  QB_CLI_S_SYNC_METHOD_GOERTZEL);
          e = QB_E_CLI;
          break;
        }
        qb->have_forced_shift = 1;
      }
      
      if ( (QB_SYNC_METHOD_GOERTZEL == qb->sync_method) && ! have_scale_2400_power ) {
        // for -s freq, if not overridden, use 2400 power scaler = 0.9 rather than 1.0
        // (empirically slightly better results)
        qb->scaler_for_2400_power = 0.9f;
      }
      
    } while (0);
    
  }
  
  return e;
  
}

static const char *phase_scheme_to_string (u8_t ps) {
  if (QB_PHASE_SCHEME_FIXED_AUTO == ps) {
    return QB_CLI_S_PHASE_SCHEME_FIXED_AUTO;
  } else if (QB_PHASE_SCHEME_FIXED_0 == ps) {
    return QB_CLI_S_PHASE_SCHEME_FIXED_0;
  } else if (QB_PHASE_SCHEME_FIXED_90 == ps) {
    return QB_CLI_S_PHASE_SCHEME_FIXED_90;
  } else if (QB_PHASE_SCHEME_FIXED_180 == ps) {
    return QB_CLI_S_PHASE_SCHEME_FIXED_180;
  } else if (QB_PHASE_SCHEME_FIXED_270 == ps) {
    return QB_CLI_S_PHASE_SCHEME_FIXED_270;
  } else if (QB_PHASE_SCHEME_BLOCK_AUTO == ps) {
    return QB_CLI_S_PHASE_SCHEME_BLOCK;
  } else {
    return "???";
  }
}


static qb_err_t parse_source_selector_simple (char *argv, qb_t *qb) {
  // deselect default 0/L
  if ( ! strcmp(argv, "L") ) {
    ;
  } else if ( ! strcmp(argv, "R") ) {
    qb->use_right_channel = 1;
  } else {
    fprintf(QB_ERR, "E: Illegal channel selection (\"%s\", expected \"L\" or \"R\").\n", argv);
    return QB_E_CLI_BAD_CHNL;
  }
  return QB_E_OK;
}



static void print_cli_summary (qb_t *qb) {
  char *sync_mode_text;
  char *ec;
  printf("Input: \"%s\"\n", qb_util_basename(qb->ipfile_path)); // 2.0.4: basename
  if (qb->using_default_op_file) {
    printf("Using default TIBET output file: %s\n", qb->tibet_opfile);
  } else if (NULL != qb->tibet_opfile) {
    printf("TIBET output to: %s\n", qb->tibet_opfile);
  }
  if (NULL != qb->tibet_z_opfile) { // 2.0.2: report multiple simultaneous filetypes
    printf("TIBETZ output to: %s\n", qb->tibet_z_opfile);
  }
  if ( NULL != qb->csw_opfile) { // 2.0.2: report multiple simultaneous filetypes
    printf("CSW output to: %s\n", qb->csw_opfile);
  }
  if (qb->find_best_hf_pwr) { // 2.0.4: scan mode
    printf("Scanning for best 2400 Hz power scaler; no output will be produced.\n");
  }

  if (qb->allow_overwrite) {
    printf("Overwriting files is permitted.\n");
  }
  if (NULL != qb->inspect.dir) {
    printf("Write inspection WAVs to: %s\n",
           qb->inspect.dir);
  }  
  sync_mode_text = "?";
  if (QB_SYNC_METHOD_PLL == qb->sync_method) {
    sync_mode_text = "PLL-based";
  } else if (QB_SYNC_METHOD_WALK == qb->sync_method) {
    sync_mode_text = "time-based";
  } else if (QB_SYNC_METHOD_GOERTZEL == qb->sync_method) {
    sync_mode_text = "frequency-based";
  }
  printf("Using %s sync generation.\n", sync_mode_text);
  if (qb->do_filtering) {
    printf("Applying bandpass pre-filter.\n");
  }
  if (qb->normalise) {
    printf("Pre-normalising input amplitude.\n");
  }
  if (QB_SYNC_METHOD_GOERTZEL != qb->sync_method) {
    printf("Using phase scheme \"%s\".\n",
           phase_scheme_to_string (qb->phase_scheme));
    if (QB_PHASE_DETECT_METHOD_PLL == qb->phase_detect_method) {
      printf("Using PLL phase detection method.\n");
    } else {
      printf("Using walk phase detection method.\n");
    }
  }
  //if (qb->no_shift_search && (QB_SYNC_METHOD_GOERTZEL != qb->sync_method)) {
  //  printf("Disabling optimal time-shift search.\n");
  //}
  if (qb->have_forced_shift) {
    printf("Forcing time-shift of %d samples.\n", qb->forced_shift);
  }
  if (qb->no_generic_odd_run_fix ) {
    if (QB_SYNC_METHOD_PLL == qb->sync_method) {
      printf("Disabling PLL atom-insertion odd-run fix.\n");
    } else if (QB_SYNC_METHOD_GOERTZEL == qb->sync_method) {
      printf("Lifting frequency-mode even-run constraint.\n");
    }
  }
  if (QB_ERRCORR_NONE != qb->err_correction_strategy) {
    ec = "BUG";
    if (QB_ERRCORR_ONE_WINS == qb->err_correction_strategy) {
      ec = "1-cycle wins";
    } else if (QB_ERRCORR_ZERO_WINS == qb->err_correction_strategy) {
      ec = "0-cycle wins";
    } else if (QB_ERRCORR_EARLY_WINS == qb->err_correction_strategy) {
      ec = "Earlier cycle wins";
    } else if (QB_ERRCORR_LATE_WINS == qb->err_correction_strategy) {
      ec = "Later cycle wins";
    } else if (QB_ERRCORR_CONFIDENCE_WINS == qb->err_correction_strategy) {
      ec = "Confident cycle wins";
    } else if (QB_ERRCORR_SINGLETON_WINS == qb->err_correction_strategy) {
      ec = "Singleton wins";
    }
    printf("Using atom flip error correction: %s", ec);
  }
  if (qb->verbose) {
    printf("Printing extra warnings.\n");
  }
  if ( qb->quieter ) {
    printf("Progress updates are disabled.\n");
  }
  if (qb->no_csw_polarity_correction) {
    printf("Omitting synthetic, MakeUEF-friendly, polarity-correcting CSW pulses.\n");
  }
  if (qb->forced_scaler_for_2400_power) {
    printf("2400 Hz power scaler set to %f.\n", qb->scaler_for_2400_power);
  }
#ifdef QB_SCHMITT_TRIGGER_THRESH_OPTIONS
  printf("Silence-on threshold power ratio:        %f\n", qb->silence_on_frac);
  printf("Signal-on threshold power ratio:         %f\n", qb->signal_on_frac);
#endif
  printf("Weak signal span hush threshold: %f\n", qb->summed_silence_on_frac);
  printf("\n");
}


static void qb_finish (qb_t *qb) {

  u8_t j;
  
  if (NULL == qb) { return; }
  
  if (NULL != qb->cwd) {
    qb_free(qb->cwd);
    qb->cwd = NULL;
  }
  
  // fclose() all inspect files
  if (qb->inspect.enabled) {
    for (j=0; j < QB_NUM_INSPECT_FILES; j++) {
      if (qb->inspect.files[j].priv.opened) { 
        qb_close_inspect_file(&(qb->inspect.files[j]));
      }
    }
  }
  
  if (qb->csw_opfile != NULL) {
    qb_free(qb->csw_opfile);
    qb->csw_opfile = NULL;
  }
  if (qb->tibet_opfile != NULL) {
    qb_free(qb->tibet_opfile);
    qb->tibet_opfile = NULL;
  }
  if (qb->tibet_z_opfile != NULL) {
    qb_free(qb->tibet_z_opfile);
    qb->tibet_z_opfile = NULL;
  }
  if (qb->inspect.dir != NULL) {
    qb_free(qb->inspect.dir);
    qb->inspect.dir = NULL;
  }
  if (NULL != qb->ipfile_buf) {
    qb_free(qb->ipfile_buf);
    qb->ipfile_buf = NULL;
  }
  if (NULL != qb->ipfile_path) {
    qb_free(qb->ipfile_path);
    qb->ipfile_path = NULL;
  }
  
  memset(qb, 0, sizeof(qb_t));
  
}




static qb_err_t main_prologue (int *argc, char *argv[], char ***my_argv_out) {

  int j;

#ifdef QB_WINDOWS
  LPWSTR win_cmdline;
  LPWSTR *win_argv;

  win_cmdline = GetCommandLineW();
  win_argv = CommandLineToArgvW(win_cmdline, argc);
#endif

  (*my_argv_out) = (char **) qb_malloc(sizeof(char *) * *argc);

  if (NULL == *my_argv_out) {
    fprintf(QB_ERR, "E: Out of memory copying command-line arguments.\n");
    return QB_E_MALLOC;
  }

  for (j = 0; j < *argc; j++) {

    int k;
    size_t size = 0;

#ifdef QB_WINDOWS
    size = WideCharToMultiByte(CP_UTF8, 0, win_argv[j], -1, NULL, 0, NULL, NULL);
#else
    size = 1 + strlen(argv[j]);
#endif
    (*my_argv_out)[j] = (char *) qb_malloc (size);

    if (NULL == (*my_argv_out)[j]) {
      fprintf(QB_ERR, "E: Out of memory copying command-line arguments.\n");
      for (k = 0; k < j; k++) {
        qb_free((*my_argv_out)[k]);
      }
      return QB_E_MALLOC;
    }

#ifdef QB_WINDOWS
    //if (size > 0x7fffffff) {
    //  return QB_E_WIN_ARGV_WCHAR_TO_MULTIBYTE;
    //}
    WideCharToMultiByte(CP_UTF8, 0, win_argv[j], -1, (*my_argv_out)[j], size & 0x7fffffff, NULL, NULL);
#else
    memcpy((*my_argv_out)[j], argv[j], size);
#endif

  }

  return QB_E_OK;

}

#ifdef QB_WINDOWS_HEAP_DEBUG
#include "crtdbg.h"
#endif

#include "vector.h"


#ifdef QB_WINDOWS
int CALLBACK WinMain(_In_ HINSTANCE hInstance,
  _In_ HINSTANCE hPrevInstance,
  _In_ LPSTR     lpCmdLine,
  _In_ int       nCmdShow) {

  int argc;
  char **argv;

  argv = NULL;

  SetConsoleOutputCP(65001);

#ifdef QB_WINDOWS_HEAP_DEBUG
  _CrtSetDbgFlag(_CRTDBG_CHECK_ALWAYS_DF | _CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF);
#endif

#else
int main (int argc, char *argv[]) {
#endif
  
  qb_t qb;
  qb_err_t e;
  char **argv2;
  int i;

  e = main_prologue (&argc, argv, &argv2);

  if (QB_E_OK != e) { return e; }

  memset(&qb, 0, sizeof(qb_t));
  
  do { // try {
  
    printf("\n%s, by 'Diminished' -- %s.\n\n",
           QB_VERSION_STRING( QB_VERSION_U32_MAJOR,
                              QB_VERSION_U32_MINOR,
                              QB_VERSION_U32_SUB ),
           QB_VERSION_DATE);
         
    // does nothing on windows
    e = qb_get_working_directory (&(qb.cwd));
    if (QB_E_OK != e) { break; }
    
    e = qb_cli (&qb, argc, argv2);
    if (QB_E_OK != e) { break; }
    
    print_cli_summary(&qb);
    
    e = check_all_filenames_are_sane(&qb);
    if (QB_E_OK != e) { break; }

    e = qb_load_ipfile (&qb);
    if (QB_E_OK != e) { break; }
    
    e = qb_process_main (&qb, argc, argv2);
    if (QB_E_OK != e) { break; }
    
  } while (0);

  for (i = 0; i < argc; i++) {
    qb_free(argv2[i]);
  }
  qb_free(argv2);
  
  qb_finish(&qb);
  
  if (QB_E_CLI != e) {
    if (QB_E_OK != e) {
      fprintf (QB_ERR, "E: Error code #%d.\n", e);
    } else {
      printf ("OK.\n");
    }
  }
 
  return e;
  
}




static qb_err_t qb_load_ipfile (qb_t *qb) {

  qb_err_t e;
    
  e = qb_read_file (qb->ipfile_path,
                    &(qb->ipfile_buf),
                    &(qb->ipfile_buf_len),
                    QB_MAX_IPFILE_LEN);
  
  return e;
  
}

#include "ctype.h"

static qb_err_t cli_parse_int (char *s, u8_t allow_negative, u8_t max_string_len, s64_t *result) {
  size_t i, len;
  len = strlen(s);
  if (len > max_string_len) {
    return QB_E_CLI;
  }
  i=0;
  *result = 0;
  if (allow_negative && (s[0] == '-')) {
    i++;
  }
  for (; i < len; i++) {
    if ( ! isdigit(s[i]) ) {
      return QB_E_CLI;
    }
  }
  *result = atoll(s);
  return QB_E_OK;
}

static qb_err_t cli_parse_decimal (char *s, u8_t allow_negative, double *result) {
  size_t i, len;
  u8_t found_point = 0;
  len = strlen(s);
  i=0;
  *result = NAN;
  if (allow_negative && (s[0] == '-')) {
    i++;
  }
  for (; i < len; i++) {
    if (s[i] == '.') {
      if (found_point) { return QB_E_CLI; } // double decimal point
      found_point = 1;
    } else if ( ! isdigit(s[i]) ) {
      return QB_E_CLI;
    }
  }
  *result = atof(s);
  return QB_E_OK;
}


static qb_err_t check_all_filenames_are_sane (qb_t *qb) {

  qb_err_t e;
  u8_t len;
  u8_t n;
  size_t z;
  //s16_t first_inspect_file;
#define QB_FN_LIST_LEN (QB_NUM_INSPECT_FILES + 10)
  char *all_filenames_list [QB_FN_LIST_LEN];
  
  e = QB_E_OK;
  //first_inspect_file = -1;
  
  // full list of all input and output filenames
  // we need space for:
  //   - all inspect files
  //   - one input file
  //   - up to three output files
  // we'll use QB_NUM_INSPECT_FILES + 10 to be sure
  
  // build lists.
  memset (all_filenames_list, 0, sizeof (char *) * QB_FN_LIST_LEN);
  //memset (file_is_output, 1, QB_FN_LIST_LEN);
  
  // i)   all input and output filenames have to be unique.
  // ii)  all output files (including inspect files) have to be able to be written.
  
  do {
  
    // have to make copies because stupid
  
    // add input file first
    z = strlen(qb->ipfile_path);
    all_filenames_list[0] = qb_malloc(z+1);
    if (NULL == all_filenames_list[0]) {
      fprintf(QB_ERR, "E: Out of memory allocating temporary input filename copy.\n");
      e = QB_E_MALLOC;
      break;
    }
    memcpy(all_filenames_list[0], qb->ipfile_path, z+1);
    len=1;
    
    // now clone output filenames
    if (NULL != qb->csw_opfile) {
      z = strlen(qb->csw_opfile);
      all_filenames_list[len] = qb_malloc(z+1);
      if (NULL == all_filenames_list[len]) {
        fprintf(QB_ERR, "E: Out of memory allocating temporary output CSW filename copy.\n");
        e = QB_E_MALLOC;
        break;
      }
      memcpy(all_filenames_list[len], qb->csw_opfile, z+1);
      len++;
    }
    
    if (NULL != qb->tibet_opfile) {
      z = strlen(qb->tibet_opfile);
      all_filenames_list[len] = qb_malloc(z+1);
      if (NULL == all_filenames_list[len]) {
        fprintf(QB_ERR, "E: Out of memory allocating temporary output TIBET filename copy.\n");
        e = QB_E_MALLOC;
        break;
      }
      memcpy(all_filenames_list[len], qb->tibet_opfile, z+1);
      len++;
    }
    
    if (NULL != qb->tibet_z_opfile) {
      z = strlen(qb->tibet_z_opfile);
      all_filenames_list[len] = qb_malloc(z+1);
      if (NULL == all_filenames_list[len]) {
        fprintf(QB_ERR, "E: Out of memory allocating temporary output TIBETZ filename copy.\n");
        e = QB_E_MALLOC;
        break;
      }
      memcpy(all_filenames_list[len], qb->tibet_z_opfile, z+1);
      len++;
    }
  
    // inspect files (if enabled)
    for (n=0; (n < QB_NUM_INSPECT_FILES) && qb->inspect.enabled; n++) {
    
      u8_t score;
      const char *code;
      
      score = 0;
      
      code = qb_inspect_filenames[n][1];
      
      // round one: sync method (also walk w/PLL phase det)
      if (code[0] == '*') {
        score++;
      } else if (    (QB_SYNC_METHOD_PLL == qb->sync_method)
           && (    (code[0] == 'P')
                || (code[0] == 'C'))) {
        score++;
      } else if (QB_SYNC_METHOD_WALK == qb->sync_method) {
        if (code[0] == 'W') {
          score++;
        } else if (    (code[0] == 'C')
                    && (QB_PHASE_DETECT_METHOD_PLL == qb->phase_detect_method)) {
          score++;
        }
      } else if (    (QB_SYNC_METHOD_GOERTZEL == qb->sync_method)
                  && (code[0] == 'F')) {
        score++;
      }
      
      // round two: phase
      
      if (code[1] == '*') {
        score++;
      } else if (    (    (QB_PHASE_SCHEME_FIXED_0    == qb->phase_scheme)
                       || (QB_PHASE_SCHEME_BLOCK_AUTO == qb->phase_scheme)
                       || (QB_PHASE_SCHEME_FIXED_AUTO == qb->phase_scheme) )
                  && (code[1] == '0') ) {
        score++;
      } else if (    (    (QB_PHASE_SCHEME_FIXED_90   == qb->phase_scheme)
                       || (QB_PHASE_SCHEME_BLOCK_AUTO == qb->phase_scheme)
                       || (QB_PHASE_SCHEME_FIXED_AUTO == qb->phase_scheme) )
                  && (code[1] == '9') ) {
        score++;
      } else if (    (    (QB_PHASE_SCHEME_FIXED_180  == qb->phase_scheme)
                       || (QB_PHASE_SCHEME_BLOCK_AUTO == qb->phase_scheme)
                       || (QB_PHASE_SCHEME_FIXED_AUTO == qb->phase_scheme) )
                  && (code[1] == '1') ) {
        score++;
      } else if (    (    (QB_PHASE_SCHEME_FIXED_270  == qb->phase_scheme)
                       || (QB_PHASE_SCHEME_BLOCK_AUTO == qb->phase_scheme)
                       || (QB_PHASE_SCHEME_FIXED_AUTO == qb->phase_scheme) )
                  && (code[1] == '2') ) {
        score++;
      }
      
      if (code[2] == '*') {
        score++;
      } else if (qb->do_filtering && (code[2] == 'F')) {
        score++;
      }
      
      //printf("%u %s\n", score, qb_inspect_filenames[n][0]);
      
      //if (score == 3) {
      //  printf("%s\n", qb_inspect_filenames[n][0]);
      //}
      
      if (3 == score) {
        // append to all filenames list; also check path is viable
        e = qb_inspect_build_and_stat_path (qb->inspect.dir,
                                            qb_inspect_filenames[n][0],
                                            &(all_filenames_list[len]));
        if (QB_E_OK != e) { break; }
        len++;
      }

    } // next inspect file
    
    if (QB_E_OK != e) { break; }
    
    // process list and fully-qualify all the pathnames
    for (n=0; n < len; n++) {
      char *tmp;
      tmp = NULL;
      e = qb_make_fully_qualified_path (all_filenames_list[n], &tmp, qb->cwd);
      qb_free(all_filenames_list[n]);
      all_filenames_list[n] = NULL;
      if (QB_E_OK != e) { break; }
      all_filenames_list[n] = tmp;
    }
    if (QB_E_OK != e) { break; }
    
    // (Unix): strip excess leading separators
#ifndef QB_WINDOWS
    for (n=0; n < len; n++) {
      char *f;
      size_t fnlen;
      f = all_filenames_list[n];
      fnlen = strlen(f);
      if ((0 == fnlen) || (QB_PATHSEP != f[0])) {
        fprintf(QB_ERR, "B: Qualified path is somehow empty or not prefixed with %c.\n", QB_PATHSEP);
        e = QB_E_BUG;
        break;
      }
      for (z=1; (z < fnlen) && (QB_PATHSEP == f[z]); z++) { }
      // this check isn't really needed:
      // a path that only consists of slashes can only
      // reference a directory, not a file. these can never come
      // from the inspect files list, because those are all hard-coded
      // filenames from within Quadbike itself. They could come from
      // a supplied input or output file, but in that case Quadbike
      // will fail anyway, because the supplied paths are directories
      // and not files. Anyway:
      if (z == fnlen) {
        fprintf(QB_ERR, "E: Path was made entirely of tin: %s\n", f);
        e = QB_E_PATH_MADE_ENTIRELY_OF_TIN;
        break;
      }
      //   ///a
      //      ^ you are here
      memmove (f, f + z - 1, 2 + fnlen - z);
    }
    if (QB_E_OK != e) { break; }
#endif
    
    // check all filenames for uniqueness
    for (n=0; n < len; n++) {
      s64_t m;
//printf("%s\n", all_filenames_list[n]);
      for (m=0; m < len; m++) {
        if (n==m) { continue; } // don't compare file against itself
#ifdef QB_MACOS
        if (0 == strcasecmp(all_filenames_list[n], all_filenames_list[m])) {
#else
        // we don't need to do case insensitive comparison on Windows,
        // since all the filenames have been passed through qb_make_fully_qualified_path(),
        // so they should be canonical, and therefore identical
        if (0 == strcmp(all_filenames_list[n], all_filenames_list[m])) {
#endif
          fprintf(QB_ERR, "E: Filename collision: %s\n", all_filenames_list[m]);
          e = QB_E_FILENAME_COLLISION;
          break;
        }
      }
      if (QB_E_OK != e) { break; }
    }
    if (QB_E_OK != e) { break; }
    
    // now check that we can actually write all the output files
    for (n=1; n < len; n++) {
      e = qb_check_can_write_file (all_filenames_list[n], qb->allow_overwrite);
      if (QB_E_OK != e) { break; }
    }
    
    // we don't bother checking whether the input file is OK,
    // because we'll try to open that in just a second.
    
  } while (0);

  // clean up
  for (n=0; n < QB_FN_LIST_LEN; n++) {
    if (NULL != all_filenames_list[n]) {
      qb_free(all_filenames_list[n]);
      all_filenames_list[n] = NULL;
    }
  }
  
  return e;
  
}





