/*
 *  Quadbike 2
 *  Copyright (C) 2026 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "build.h"

#include "sync_walk.h"
#include "fir.h"
#include "util.h"
#include "qbio.h" // for --inspect-dir
#include "inspect.h"
#include "errcorr.h" // for walk mode no goertzel
#include "fir_vec2.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

static qb_err_t walk_span_cycs_interpolate_sync  (qb_span_t *span,
                                                   s32_t rate,
                                                   //s32_t span_ix,
                                                   s32_t *num_synthetic_cycles_out,
                                                   double *synthetic_fraction_out);

static qb_err_t walk_append_feature (qb_wave_feature_t **featrs_inout,
                                     s64_t *featrs_alloc_inout,
                                     s64_t *featrs_fill_inout,
                                     qb_wave_feature_t *f);
                                     
static void walk_identify_feature_2 (float a,
                                     float b,
                                     float c,
                                     qb_wave_feature_t *f);





#define QB_SYNC_WALK_CYCLE_LEN_TOLERANCE_FRAC     0.15f

#define QB_SYNC_WALK_FEATURE_TEST_NOPE  0
#define QB_SYNC_WALK_FEATURE_TEST_1200  1
#define QB_SYNC_WALK_FEATURE_TEST_2400  2

// don't allow the live cycle lengths to move
// outside +/- 25% of the nominal (nominal for 2400 Hz)
#define QB_SYNC_WALK_SPEED_MIN_2400_CYC_FRAC    0.15

//qb_err_t qb_walk_identify_features (double *src_d,
qb_err_t qb_walk_identify_features (float *src_f,
                                    s64_t srclen,
                                    qb_span_t *spans,
                                    s32_t num_spans,
                                    qb_wave_feature_t **featrs_out,
                                    s64_t *num_featrs_out) {
//                                    float rate) {
  
  s64_t n;
  qb_err_t e;
  s64_t featrs_alloc;
  s64_t featrs_fill;
  qb_wave_feature_t ftr_prev;
  u64_t adjacent_identical;
  s32_t cur_span_ix;
   
  featrs_alloc = 0;
  featrs_fill = 0;
  *featrs_out = NULL;
  *num_featrs_out = 0;
  cur_span_ix = 1;

  e = QB_E_OK;
  
  memset(&ftr_prev, 0, sizeof(qb_wave_feature_t));
  
  adjacent_identical = 0;
  
  spans[0].first_feature_ix = 0;
  
  for (n=1; n < (srclen-1); n++) {
  
    qb_wave_feature_t ftr_o, ftr_o2;

    // if two adjacent sample values are the same, ignore this
    // sample for the purposes of feature detection
    if (src_f[n-1] == src_f[n]) {
      adjacent_identical++;
      continue;
    }
  
    ftr_o.sample_num = n;
    walk_identify_feature_2 (src_f[n-1],
                             src_f[n],
                             src_f[n+1],
                             &ftr_o);
    ftr_o.sample_num = n;

    if (QB_SYNC_WALK_FEATURE_MINIMUM == ftr_o.value) {
      ftr_o2 = ftr_o;
      e = walk_append_feature (featrs_out, &featrs_alloc, &featrs_fill, &ftr_o2);
      if (QB_E_OK != e) { break; } // return e; }
    } else if (QB_SYNC_WALK_FEATURE_MAXIMUM == ftr_o.value) {
      ftr_o2 = ftr_o;
      e = walk_append_feature (featrs_out, &featrs_alloc, &featrs_fill, &ftr_o2);
      if (QB_E_OK != e) { break; } //return e; }
    } else {
      continue;
    }
    
//fprintf(stderr, "cur_span_ix = %d, type = %s\n", cur_span_ix, qb_span_get_type_text(spans + cur_span_ix)); fflush(stderr);
    
    // if this was the first feature on this span, make a note of that
    if ((cur_span_ix < num_spans) && (n >= spans[cur_span_ix].start)) {
      spans[cur_span_ix].first_feature_ix = featrs_fill - 1;
      cur_span_ix++;
    }
    
#ifdef QB_FEATURE_GAP
    n++; // ensure a gap between detected features; will fail at 8KHz though!
#endif
   
    ftr_prev = ftr_o;

  } // next sample
  
  if (QB_E_OK != e) {
    qb_free(*featrs_out);
    *featrs_out = NULL;
    return e;
  }
  
  if (adjacent_identical > 0) {
    fprintf(QB_ERR, "    W: Ignored %llu identical adjacent samples.\n", adjacent_identical);
  }

  printf("    Decoded %lld features.\n", featrs_fill);
  
  *num_featrs_out = featrs_fill;
  
  return e;
  
}


#define QB_WALK_MIN_FEATURE_AMPLITUDE 0.01

// takes strictly three samples
//static void walk_identify_feature_2 (double a,
//                                     double b,
//                                     double c,
static void walk_identify_feature_2 (float a,
                                     float b,
                                     float c,
                                     qb_wave_feature_t *f) {
                                     
  //double amplitude;
  float amplitude;
                                     
  f->value = QB_SYNC_WALK_FEATURES_NONE;
  f->sample_num = -1;
  
  //amplitude = fabs(b);
  amplitude = fabsf(b);
  
  if (amplitude < QB_WALK_MIN_FEATURE_AMPLITUDE) {
    // if the amplitude of the peak is too low, then
    // we don't count it, and instead we expect
    // the interpolator (later on) to fill in the gap
    return;
  }
  
  // detect turning points
  
  if ((a==b) && (b==c)) {
    //fprintf(QB_ERR, "W: Attempting to identify feature: three identical samples\n");
    return;
  }
  
  if ((b>0.0f) && ( ((a<=b) && (c<b)) || ((a<b) && (c==b)) ) ) {
  //if ((b>0.0) && ( ((a<=b) && (c<b)) || ((a<b) && (c==b)) ) ) {
    f->value = QB_SYNC_WALK_FEATURE_MAXIMUM;
    f->amplitude = fabs(b);
  } else if ((b<0.0f) && ( ((a>=b) && (c>b)) || ((a>b) && (c==b)) ) ) {
  //} else if ((b<0.0) && ( ((a>=b) && (c>b)) || ((a>b) && (c==b)) ) ) {
    f->value = QB_SYNC_WALK_FEATURE_MINIMUM;
    f->amplitude = fabs(b);
  }

    
}


#define QB_SYNC_WALK_FEATURES_ALLOC_DELTA 100000

static qb_err_t walk_append_feature (qb_wave_feature_t **featrs_inout,
                                     s64_t *featrs_alloc_inout,
                                     s64_t *featrs_fill_inout,
                                     qb_wave_feature_t *f) {
                                     
  qb_wave_feature_t *featrs_new;
  
  if ((*featrs_fill_inout) >= (*featrs_alloc_inout)) {
    *featrs_alloc_inout += 1 + QB_SYNC_WALK_FEATURES_ALLOC_DELTA;
    featrs_new = qb_realloc (*featrs_inout,
                          sizeof(qb_wave_feature_t) * *featrs_alloc_inout);
    if (NULL == featrs_new) {
      fprintf(QB_ERR, "E: Out of memory appending feature.\n");
      return QB_E_MALLOC;
    }
    *featrs_inout = featrs_new;
  }
  (*featrs_inout)[*featrs_fill_inout] = *f;
  (*featrs_fill_inout)++;
  
  return QB_E_OK;

}


qb_err_t qb_walk_compute_gaps (qb_gaps_t *gaps,
                               float tape_speed,
                               float rate) {
  
  //double ideal_2400, ideal_1200;
  //double tol;
  float ideal_2400, ideal_1200;
  float tol;
  
//~ printf("WCG: live_1200 = %lf, live_2400 = %lf\n", cyc_len_smps_live_1200, cyc_len_smps_live_2400);

  //gaps->ideal_1200 = cyc_len_smps_live_1200 / 4.0;
  //gaps->ideal_2400 = cyc_len_smps_live_2400 / 4.0;
  
  // ideal half-2400-cycle (2400-pulse) lengths
  ideal_2400 = rate / (QB_FREQ_2 * 2.0f * tape_speed);
  ideal_1200 = ideal_2400 * 2.0f;
  
  gaps->ideal_2400_f = ideal_2400;
  gaps->ideal_2400_i = (s64_t) round(ideal_2400);
  
//  tol_smps_crit     = ideal_2400 * QB_SYNC_WALK_CRIT_GAP_TOL_2400_FRAC;
//  tol_smps_non_crit = ideal_2400 * QB_SYNC_WALK_NON_CRIT_GAP_TOL_2400_FRAC;
  
  /*
#ifdef QB_SANITY
  if (tol_smps_crit < 0.0) {
    fprintf(QB_ERR, "B: qb_walk_compute_gaps: tol_smps_crit swung negative!\n");
    return QB_E_BUG;
  }
  if (tol_smps_non_crit < 0.0) {
    fprintf(QB_ERR, "B: qb_walk_compute_gaps: tol_smps_non_crit swung negative!\n");
    return QB_E_BUG;
  }
#endif
*/

  // critical length at which a fast 1200 cycle becomes a slow 2400 one
  //gaps->thresh = (s64_t) round (1.5 * ideal_2400);
  gaps->thresh = (s64_t) roundf (1.5f * ideal_2400);
  
  // sane outside range
  // longest valid 1200 cycle or shortest valid 2400 cycle
  // feature pair gaps falling outside this range should just be disregarded
 // tol = 0.2 * ideal_2400;
  tol = 0.2f * ideal_2400;
  gaps->max_valid_1200   = (s64_t) round (ideal_1200 + tol);
  gaps->min_valid_2400_i = (s64_t) round (ideal_2400 - tol);
  gaps->min_valid_2400_f = ideal_2400 - tol;
  
  
  //gaps->thresh_one_half_to_one     = (s64_t) round((ideal_2400 * 3.0) / 4.0);
  //gaps->thresh_one_to_three_halves = (s64_t) round((ideal_2400 * 5.0) / 4.0);
  //gaps->thresh_three_halves_to_two = (s64_t) round((ideal_2400 * 7.0) / 4.0);
  gaps->thresh_one_half_to_one     = (s64_t) roundf((ideal_2400 * 3.0f) / 4.0f);
  gaps->thresh_one_to_three_halves = (s64_t) roundf((ideal_2400 * 5.0f) / 4.0f);
  gaps->thresh_three_halves_to_two = (s64_t) roundf((ideal_2400 * 7.0f) / 4.0f);
  
  /*
printf("ideal_2400                 = %lld\n", gaps->ideal_2400_i);
printf("ideal_1200                 = %lld\n", (s64_t) round(ideal_1200));
printf("thresh_one_half_to_one     = %lld\n", gaps->thresh_one_half_to_one);
printf("thresh_one_to_three_halves = %lld\n", gaps->thresh_one_to_three_halves);
printf("thresh_three_halves_to_two = %lld\n", gaps->thresh_three_halves_to_two); */
  
  return QB_E_OK;
  
}



//static u8_t is_one_half_gap (s64_t gap, qb_gaps_t *gaps) {
//  return (gap < gaps->thresh_one_half_to_one);
//}

static u8_t is_one_gap (s64_t gap, qb_gaps_t *gaps) {
  return (gap >= gaps->thresh_one_half_to_one) && (gap < gaps->thresh_one_to_three_halves);
}

static u8_t is_three_halves_gap (s64_t gap, qb_gaps_t *gaps) {
  return (gap >= gaps->thresh_one_to_three_halves) && (gap < gaps->thresh_three_halves_to_two);
}

static u8_t is_two_gap (s64_t gap, qb_gaps_t *gaps) {
  return (gap >= gaps->thresh_three_halves_to_two);
}

// phase 0 and 180 are identically gapped, but 0 starts on a peak and 180 starts on a valley
static u8_t is_phase_0 (s64_t gap1, s64_t gap2, s64_t gap3, qb_gaps_t *gaps) {
  return is_two_gap(gap1, gaps) && is_three_halves_gap(gap2, gaps) && is_one_gap(gap3, gaps);
}

static u8_t is_phase_180 (s64_t gap1, s64_t gap2, s64_t gap3, qb_gaps_t *gaps) {
  return is_two_gap(gap1, gaps) && is_three_halves_gap(gap2, gaps) && is_one_gap(gap3, gaps);
}

// similarly, 90 and 270 are identically gapped, but 90 starts on a valley and 270 starts on a peak
static u8_t is_phase_90 (s64_t gap1, s64_t gap2, qb_gaps_t *gaps) {
  return is_two_gap(gap1, gaps) && is_one_gap(gap2, gaps);
}

static u8_t is_phase_270 (s64_t gap1, s64_t gap2, qb_gaps_t *gaps) {
  return is_two_gap(gap1, gaps) && is_one_gap(gap2, gaps);
}


void qb_get_data_span_phases_by_feats  (qb_span_t *spans,
                                        s32_t num_spans,
                                        qb_wave_feature_t *feats,
                                        s64_t num_feats,
                                        float rate,
                                        u8_t dp,   // show percentage updates?
                                        u8_t silent) { // 2.0.4: completely silent

  s32_t sn;
  
  if ( ! silent ) {
    printf("    Measuring span phases (walk method): ");
    fflush(stdout); // MacOS
    qb_show_meter(dp);
  }
  
  for (sn=0; sn < num_spans; sn++) {
  
    qb_span_t *span;
    s64_t fn;
    qb_gaps_t gaps;
    s64_t scores[4];
    u8_t phix;
    s64_t best_score;
    u8_t best_phix;
    
    memset(scores, 0, 4 * sizeof(s64_t));
    
    span = spans + sn;
    
    if (QB_SPAN_TYPE_DATA != span->type) {
      continue;
    }
    
    qb_walk_compute_gaps (&gaps, span->speed, rate);
    
    for (fn=0; fn < (num_feats - 3); fn++) {
    
      qb_wave_feature_t *feat;
      
      feat = feats + fn;
      
      if (    (QB_SYNC_WALK_FEATURE_MAXIMUM == feat->value)
           //&& (QB_SYNC_WALK_FEATURE_MINIMUM == (feat+1)->value)
           //&& (QB_SYNC_WALK_FEATURE_MAXIMUM == (feat+2)->value)
           //&& (QB_SYNC_WALK_FEATURE_MINIMUM == (feat+3)->value)
           && is_phase_0 ((feat+1)->sample_num - feat->sample_num,
                          (feat+2)->sample_num - (feat+1)->sample_num,
                          (feat+3)->sample_num - (feat+2)->sample_num,
                          &gaps)) {
        scores[0]++;
      } else if (    (QB_SYNC_WALK_FEATURE_MINIMUM == feat->value)
                  //&& (QB_SYNC_WALK_FEATURE_MAXIMUM == feat->value)
                  //&& (QB_SYNC_WALK_FEATURE_MINIMUM == feat->value)
                  && is_phase_90 ((feat+1)->sample_num - feat->sample_num,
                                  (feat+2)->sample_num - (feat+1)->sample_num,
                                  &gaps)) {
        scores[1]++;
      } else if (    (QB_SYNC_WALK_FEATURE_MINIMUM == feat->value)
                  //&& (QB_SYNC_WALK_FEATURE_MAXIMUM == (feat+1)->value)
                  //&& (QB_SYNC_WALK_FEATURE_MINIMUM == (feat+2)->value)
                  //&& (QB_SYNC_WALK_FEATURE_MAXIMUM == (feat+3)->value)
                  && is_phase_180 ((feat+1)->sample_num - feat->sample_num,
                                   (feat+2)->sample_num - (feat+1)->sample_num,
                                   (feat+3)->sample_num - (feat+2)->sample_num,
                                   &gaps)) {
        scores[2]++;
      } else if (    (QB_SYNC_WALK_FEATURE_MAXIMUM == feat->value)
                  //&& (QB_SYNC_WALK_FEATURE_MINIMUM == feat->value)
                  //&& (QB_SYNC_WALK_FEATURE_MAXIMUM == feat->value)
                  && is_phase_270 ((feat+1)->sample_num - feat->sample_num,
                                   (feat+2)->sample_num - (feat+1)->sample_num,
                                   &gaps)) {
        scores[3]++;
      }
    } // next feature
//printf("span %d: scores %lld, %lld, %lld, %lld\n", sn, scores[0], scores[1], scores[2], scores[3]);
    for (phix=0, best_phix=0, best_score=0; phix < 4; phix++) {
      if (scores[phix] > best_score) {
        best_score = scores[phix];
        best_phix = phix;
      }
    }
    span->detected_input_phase_ix = best_phix;
    if ( ! silent ) {
      qb_update_meter (dp, sn, num_spans, 1.0f, 0);
    }
  } // next span
  
  if ( ! silent ) {
    qb_hide_meter (dp, 0);
    printf("done.\n");
  }

}





qb_err_t qb_sync_walk  (qb_wave_feature_t *feats,
                        s64_t num_feats,
                        s32_t rate,
                        qb_span_t *spans,
                        s32_t num_spans) {

  qb_err_t e;
  s32_t sn;

  // 2.0.4: sanity
#ifdef QB_SANITY
  if (NULL == feats) {
    fprintf(QB_ERR, "B: qb_sync_walk: feats is NULL\n");
    return QB_E_BUG;
  }
#endif
  
  e = QB_E_OK;
  
  for (sn=0; sn < num_spans; sn++) {
  
    qb_span_t *span;
    qb_gaps_t gaps;
    s64_t feats_in_this_span;
    
    span = spans + sn;
    
    // data spans only
    if (QB_SPAN_TYPE_DATA != span->type) {
      continue;
    }
    // ******************************
    // 3. WORK OUT VARIOUS GAP RANGES for span
    // ******************************
    e = qb_walk_compute_gaps (&gaps, span->speed, (float) rate);
    if (QB_E_OK != e) { return e; }
    
    /*
    printf("SPAN %d half-cycle gaps (num. samples):\n"
           "  ideal 2400:       %lld (%lf)\n"
           "  sane range:       %lld (%lf) -> %lld\n"
           "  1200/2400 thresh: %lld\n"
           "  ambiguous region: %lld -> %lld\n",
           sn,
           gaps.ideal_2400_i, gaps.ideal_2400_f,
           gaps.min_valid_2400_i, gaps.min_valid_2400_f, gaps.max_valid_1200,
           gaps.thresh,
           gaps.max_2400_polarity_testing, gaps.min_1200_polarity_testing);
    */
    
    // get number of features in this span
    // (for final span, use total features as upper bound)
    
    if (sn < (num_spans - 1)) {
      feats_in_this_span = ((span+1)->first_feature_ix) - (span->first_feature_ix);
    } else {
      feats_in_this_span = num_feats - (span->first_feature_ix);
    }
    
    e = qb_sync_walk_features_to_cycles (feats + span->first_feature_ix,
                                         feats_in_this_span,
                                         rate,
                                         span->speed,
                                         span->detected_input_phase_ix,
                                         &(span->atoms),
                                         &(span->num_atoms));
    if (QB_E_OK != e) { break; }
  }
  
  return e;
  
}


qb_err_t qb_spans_walk_interp_absent_cycs  (qb_inspect_t *inspect,
                                            qb_span_t *spans,
                                            s32_t num_spans,
                                            s32_t rate) {

  qb_err_t e;
  s32_t n;
  s64_t prev_smpnum;
  
  e = QB_E_OK;

  // before interpolating, write out uninterpolated inspection cycles
  if (inspect->enabled) {
    prev_smpnum = 0;
    for (n=0; n < num_spans; n++) {
      qb_span_t *span0, *span1;
      s64_t prev_smpnum_sanity_check;
      span1 = NULL;
      span0 = (spans) + n;
      if (n < (num_spans - 1)) {
        span1 = span0 + 1;
      }
      prev_smpnum_sanity_check = prev_smpnum;
      e = qb_inspect_append_bitsamples (span0,
                                        span1, // or NULL
                                        &prev_smpnum, // updated
                                        n,
                                        //inspect,
                                        inspect->files + QB_INSPECT_FILE_IX_WALK_UNINTERPOLATED);
      if (QB_E_OK != e) { break; }
      // sanity
      if (prev_smpnum < prev_smpnum_sanity_check) {
        fprintf(QB_ERR, "B: %s: Catastrophic: prev_smpnum shrank in qb_inspect_append_bitsamples call (%lld -> %lld)?!\n",
                QB_FUNC_M, prev_smpnum_sanity_check, prev_smpnum);
        e = QB_E_BUG;
        break;
      }
    }
  }
  
  if (QB_E_OK != e) { return e; }

  // now do the interpolation
  for (n=0; n < num_spans; n++) {
    qb_span_t *span;
    s32_t num_synthetic_cycs_generated;
    double synthetic_fraction;
    span = spans + n;
    if (QB_SPAN_TYPE_DATA != span->type) {
      // consider data spans only
      continue;
    }
    if (span->num_atoms < 2) {
      fprintf(QB_ERR, "    W: [%lld] interp. walk cycs: %s span #%d (len %lld) has <2 cycs! skip\n",
              qb_get_span_start_accurate_or_rough(span), qb_span_get_type_text(span), n, span->len);
      continue;
    }
    num_synthetic_cycs_generated = 0;
    synthetic_fraction = NAN;
    e = walk_span_cycs_interpolate_sync (span,
                                         rate,
                                         //n,
                                         &num_synthetic_cycs_generated,
                                         &synthetic_fraction);
    if (QB_E_OK != e) {
      fprintf(QB_ERR, "E: [%lld] qb_walk_span_cycs_interpolate_sync failed on %s span #%d (len %lld)\n",
              qb_get_span_start_accurate_or_rough(span), qb_span_get_type_text(span), n, span->len);
      break;
    }
    //total_synth_cycs += num_synthetic_cycs_generated;
    // in a perfect world in which walk mode inserted all the needed cycles
    // (it doesn't any more because of the lame phase implementation)
    // we'd check synthetic_fraction here and if it were > 0.1 or so,
    // we'd increment a counter that counts the number of spans with
    // high synthetic cycle counts and print a warning, but since interpolation is now
    // a vital part of creating cycles for walk mode, we can't do this
    // until walk mode cycle generation is rewritten to behave itself for all phases
    // (currently it relies on interpolation for 90 and 270 phases)
  }
  
  return e;
  
}


static qb_err_t walk_span_cycs_interpolate_sync  (qb_span_t *span,
                                                   s32_t rate,
                                                   //s32_t span_ix,
                                                   s32_t *num_synthetic_cycles_out,
                                                   double *synthetic_fraction_out) {

  s32_t i;
  qb_gaps_t gaps;
  qb_err_t e;
  qb_atom_t *cycs2;
  s32_t alloc, fill;
  double thresh_gap_smps;
  s32_t total_new_cycles;
  s64_t prev_smpnum;
  
  *num_synthetic_cycles_out = 0;
  *synthetic_fraction_out = NAN;
  
  if (span->num_atoms < 2) {
    fprintf (QB_ERR,
             "B: %s: num_cycs_inout is < 2 (%d)\n",
             QB_FUNC_M, span->num_atoms);
    return QB_E_BUG;
  }
  
  alloc = span->num_atoms; // *num_cycs_inout;
  
//~ qb_cycles_check_integrity (*cycs_inout, *num_cycs_inout);
  
  cycs2 = qb_malloc (sizeof(qb_atom_t) * alloc);
  if (NULL == cycs2) {
    fprintf(QB_ERR, "E: Out of memory allocating cycle interpolation buffer.\n");
    return QB_E_MALLOC;
  }
  
  fill = 0;
  total_new_cycles = 0;
  
//~ printf("cycles_interpolate_sync: num_cycs = %d\n", *num_cycs_inout);

  // compute min/ideal/max timings for various things:
  e = qb_walk_compute_gaps (&gaps,
                            span->speed,
                            (float) rate);
  if (QB_E_OK != e) {
    qb_free(cycs2);
    return e;
  }
  
  // threshold for detecting one or more missed cycles
  // if the gap between cycles is longer than twice the minimum 2400 cycle len
  thresh_gap_smps = round (2.0 * 2.0 * gaps.min_valid_2400_f);
  
  prev_smpnum = span->atoms[0].sample_num;

  for (i=1; i < span->num_atoms; i++) {
  
    qb_atom_t c, old;
    s32_t num_cycs_to_add;
    s32_t gap_len_in_cycs;
    s32_t m;
    s64_t added_cycles_len_smps;
    double added_cycles_len_smps_f;
    s64_t gap_smps;
    s64_t samples_remaining;

    //gap_smps = (*cycs_inout)[i].samples_since_last;
    gap_smps = span->atoms[i].sample_num - prev_smpnum;
    
#ifdef QB_SANITY
    if (gap_smps < 1) {
      fprintf(QB_ERR, "B: %s: gap_smps is bad (%lld)\n", QB_FUNC_M, gap_smps);
      e = QB_E_BUG;
      break;
    }
#endif
    
//printf("gap_smps = %lld, thresh_gap_smps = %lf\n",
//       gap_smps, thresh_gap_smps);
    
    // if detected gap is more than double the minimum gap,
    // then there is space to insert at least one new cycle
    if (gap_smps > thresh_gap_smps) {
    
      // so we have some empty space, and we want to cut it up into new cycles
      // if ssl is, let's say, smaller than 2 * ideal, then maybe
      // (ssl / ideal) is something like 1.8, in which case we
      // want to insert a new cycle that gives two x 0.9 gaps
      
      gap_len_in_cycs = ((s32_t) round (gap_smps / (2.0 * gaps.ideal_2400_f)));
      num_cycs_to_add = gap_len_in_cycs - 1;

//printf("interpolate: gap = %lld, thresh = %lf, ideal = %lf, num_cycs_to_add = %d\n",
//       gap_smps, thresh_gap_smps, 2 * gaps.ideal_2400_f, num_cycs_to_add);
      
      if (num_cycs_to_add > 30) {
        // don't fill in enormous gaps;
        // just copy the original cycle over
        e = qb_append_atom (span->atoms + i, &cycs2, &alloc, &fill);
        prev_smpnum = span->atoms[i].sample_num;
        continue;
      }
      
      // divide the available space up equally:
      added_cycles_len_smps_f = (gap_smps / (double) gap_len_in_cycs);
      
      added_cycles_len_smps = (s64_t) round(added_cycles_len_smps_f);
      
      //~ printf ("[%10lld] interp: sync absent for %lld samples; insert %d new of len %lf (%lld)\n",
              //~ (*cycs_inout)[i].sample_num,
              //~ gap_smps,
              //~ num_cycs_to_add,
              //~ added_cycles_len_smps_f,
              //~ added_cycles_len_smps);

      // this is the time period we need to consume
      //~ samples_remaining_f = gap_smps;
      samples_remaining = gap_smps;
//~ printf("added_cycles_len_smps = %lld\n", added_cycles_len_smps);
      
      // synthetic cycles are placed *before*
      // the original one

      if (i > 0) {
        old = span->atoms[i - 1];
      } else {
        qb_init_atom(&old);
      }
      
//printf("interpolate: ");
      
      // create all-new synthetic cycles!
//~ printf("adding %d before [%lld]\n", num_cycles_to_add, (*cycs_inout)[i].sample_num);
      for (m=0; m < num_cycs_to_add; m++) {
      
        qb_atom_t c2;
        
        // use the previous cycle as a template
        // for the new, synthetic ones
        
        c2 = old;
        
        c2.sample_num += added_cycles_len_smps;
        //c2.samples_since_last = added_cycles_len_smps;
        c2.synthetic = 1;
       //c2.walk_value = 0;
        
//printf("%lld, ", c2.sample_num);
        
        e = qb_append_atom (&c2, &cycs2, &alloc, &fill);
        if (QB_E_OK != e) { break; }
        
        old = c2;
        //~ samples_remaining_f -= added_cycles_len_smps_f;
        samples_remaining -= added_cycles_len_smps;
        
      } // next interpolated cycle to be inserted
      
//printf("\n");
      
      if (QB_E_OK != e) { break; }
      
      total_new_cycles += num_cycs_to_add;
      
      // modify original cycle (i.e. the latest one, chronologically)
      // sample number remains unchanged, since it doesn't move, but
      // samples_since_last is now wrong and must be fixed up
      c = span->atoms[i];
      
#ifdef QB_SANITY
      //~ if (samples_remaining_f < 0.0) {
      if (samples_remaining < 0) {
        //~ fprintf(QB_ERR, "B: cycles_interpolate_sync: samples_remaining < 0.0 (%lf)!\n",
                //~ samples_remaining_f);
        fprintf(QB_ERR, "B: %s: samples_remaining < 0 (%lld)!\n",
                QB_FUNC_M, samples_remaining);
        e = QB_E_BUG;
        break;
      }
#endif
      
      e = qb_append_atom (&c, &cycs2, &alloc, &fill);

    } else {
      // just copy the original cycle over
      e = qb_append_atom(span->atoms + i, &cycs2, &alloc, &fill);
    }
    
    prev_smpnum = span->atoms[i].sample_num;
    
    if (QB_E_OK != e) { break; }
    
  } // next cycle
  
  //if ((QB_E_OK == e) && (((total_new_cycles * 100.0f) / fill) > 1.0)) {
  //  fprintf(QB_ERR, "    W: [%lld] span #%d: excessive synthetic cycles (%d/%d, %.1f%%).\n",
  //          qb_get_span_start_accurate_or_rough(span), span_ix, total_new_cycles, fill, (total_new_cycles * 100.0f) / fill);
  //}
  
  *num_synthetic_cycles_out = total_new_cycles;
  *synthetic_fraction_out = total_new_cycles / fill;
  
  if (QB_E_OK != e) {
    qb_free(cycs2);
  } else {
    // replace inputted cycles list with replacement one
    qb_free(span->atoms);
    span->atoms = cycs2;
    span->num_atoms = fill;
  }

//~ printf("REMOVE ME: doing cycles integrity check\n");
//~ if (QB_E_OK == e) {
  //~ e = qb_cycles_check_integrity(*cycs_inout, *num_cycs_inout);
//~ }
  
  return e;
  
}




qb_err_t qb_sync_walk_features_to_cycles (qb_wave_feature_t *features,
                                          s64_t num_features,
                                          s32_t sample_rate,
                                          float tape_speed,
                                          u8_t phase_ix, // alpha-2
                                          qb_atom_t **cycles_out,
                                          s32_t *num_cycles_out) {
  
  qb_err_t e;
  s64_t n, m;
  s32_t cycs_alloc;
  qb_gaps_t gaps;
  u8_t first, second;
  
  *cycles_out = NULL;
  *num_cycles_out = 0;
  cycs_alloc = 0;
  
  // recompute gap lengths based on live tape speed
  e = qb_walk_compute_gaps (&gaps,
                            tape_speed,
                            (float) sample_rate);
  if (QB_E_OK != e) { return e; }
  
  // OK
  // for phases 0, 90, 270, we want (peak, valley)
  // for phase 180 we want (valley, peak)
    
  if (2 == phase_ix) {
    first  = QB_SYNC_WALK_FEATURE_MINIMUM;
    second = QB_SYNC_WALK_FEATURE_MAXIMUM;
  } else {
    first  = QB_SYNC_WALK_FEATURE_MAXIMUM;
    second = QB_SYNC_WALK_FEATURE_MINIMUM;
  }

  for (n=0; n < (num_features - 1); n++) {
  
    qb_atom_t cyc;
    //s64_t smpnum;
    qb_wave_feature_t *twf0;
    qb_wave_feature_t *twf1;
    s64_t gap;
    
    u8_t is_long_cycle;
    s64_t cycpos1, cycpos2;

    twf0 = features + n;
    
    //cyc.orig_sample_num = -1; // FIXME: eliminate this garbage


    // the order must be correct
    // if it's the other way round then continue with next feature
    if (first != twf0->value) {
      continue;
    }
    
    // (start at next consecutive feature)
    for (m = n + 1; m < (num_features - 1); m++) {
      twf1 = features + m;
      if (second == twf1->value) {
        break;
      }
    }
    
    twf1 = features + m; // shut compiler up
    
    gap = (twf1->sample_num - twf0->sample_num);
    
#ifdef QB_SANITY
    if (gap < 0) {
      fprintf(QB_ERR, "B: %s: gap < 0\n", QB_FUNC_M);
      e = QB_E_BUG;
      break;
    }
#endif

    is_long_cycle = 0;
    
    // we shouldn't experience any 3/2 gaps, with the way
    // we're measuring things.
    
    // is the gap sane?
    if ((gap > gaps.min_valid_2400_i) && (gap < gaps.max_valid_1200)) {
      // yes, compare to threshold
      if (gap >= gaps.thresh) {
        // 1200
        is_long_cycle = 1;
      }
    } else {
      // invalid gap; just move to next feature
      // ??? advance n past m? hmm ???
      continue;
    }
    
    qb_init_atom(&cyc);
    
    cycpos1 = -1;
    cycpos2 = -1;
    
    if ((0 == phase_ix) || (2 == phase_ix)) {
      // phase 0 or 180
      // for 0 it goes (peak, valley); for 180 it goes (valley, peak)
      // otherwise they are identical
      if (is_long_cycle) {
        // long gap
        // sample at peak and valley
        cycpos1 = twf0->sample_num;
        cycpos2 = twf1->sample_num;
      } else {
        // short gap
        // sample in middle
        cycpos1 = twf0->sample_num + ((twf1->sample_num - twf0->sample_num) / 2);
      }
    } else if (1 == phase_ix) {
      // phase 90
      if (is_long_cycle) {
        // long gap
        // sample in middle
        cycpos1 = twf0->sample_num + ((twf1->sample_num - twf0->sample_num) / 2);
      } else {
        // short gap
        // sample at end
        cycpos1 = twf1->sample_num;
      }
      // note that we miss a sampling instant on long cycle's valley->peak
      // we'll need the interpolation to insert this for us
    } else {
      // phase 270
      if (is_long_cycle) {
        // long gap
        // sample in middle
        cycpos1 = twf0->sample_num + ((twf1->sample_num - twf0->sample_num) / 2);
      } else {
        // short gap
        // sample at start
        cycpos1 = twf0->sample_num;
      }
      // note that we miss a sampling instant on long cycle's valley->peak
      // we'll need the interpolation to insert this for us
    }
    
    cyc.sample_num = cycpos1;
    e = qb_append_atom (&cyc, cycles_out, &cycs_alloc, num_cycles_out);
    if (QB_E_OK != e) { break; }
    
    if (cycpos2 != -1) {
      cyc.sample_num = cycpos2;
      e = qb_append_atom(&cyc, cycles_out, &cycs_alloc, num_cycles_out);
      if (QB_E_OK != e) { break; }
    }
    
    n = m;

  }
  
//~ qb_cycles_check_integrity(*cycles_out, *num_cycles_out);
  
  if (QB_E_OK != e) {
    // clean up on error
    qb_free(*cycles_out);
    *cycles_out = NULL;
    *num_cycles_out = 0;
  }

  return e;

}


// "trad mode"
/*
qb_err_t walk_assign_cyc_values_by_cyc_lens_no_goertzel (qb_source_t *sources,
                                                         u8_t num_sources,
                                                         u8_t err_correction_strategy) {
  u8_t src_n;
  qb_err_t e;
  e = QB_E_OK;
  for (src_n=0; src_n < num_sources; src_n++) {
    qb_source_t *src;
    s32_t sn;
    src = sources + src_n;
    if (src->disabled) { continue; }
    for (sn=0; sn < src->num_spans; sn++) {
      qb_span_t *span;
      s32_t an;
      span = src->spans + sn;
      for (an=0; an < span->num_atoms; an++) {
        qb_atom_t *cyc;
        cyc = span->atoms + an;
        if (QB_SPAN_TYPE_LEADER == span->type) {
          cyc->value = 'L';
        } else if (QB_SPAN_TYPE_SILENT == span->type) {
          cyc->value = 'S';
        } else if (cyc->walk_value == 1) {
          cyc->value = '1';
        } else {
          cyc->value = '0';
        }
          
      }
    } // next span
    
    if (err_correction_strategy != QB_ERRCORR_NONE) {
      e = qb_do_atomflip_correction_all_spans (src->spans, src->num_spans, err_correction_strategy);
    }
    if (QB_E_OK != e) { break; }
    
  } // next src

  return e;
  
}
*/

