/*
 *  Quadbike 2
 *  Copyright (C) 2026 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "tibet.h"
#include "span.h"
#include "process.h"
#include "qbio.h"

#include <stdlib.h>
#include <string.h>

#define QB_TIBET_VERSION "0.5" // QB 2.0.2

// keywords starting with / are "hints";
// they contain helpful information, but may be
// (and probably should be) ignored by decoders

#define QB_TIBET_S_VERSION "tibet"
#define QB_TIBET_S_TIME    "/time"
#define QB_TIBET_S_SPEED   "/speed"
#define QB_TIBET_S_PHASE   "/phase"
#define QB_TIBET_S_LEADER  "leader"
#define QB_TIBET_S_SILENCE "silence"
#define QB_TIBET_S_SQUAWK  "squawk"
#define QB_TIBET_S_DATA    "data"
#define QB_TIBET_S_END     "end"

qb_err_t qb_write_tibet (qb_span_t *spans,
                         s32_t num_spans,
                         s32_t rate,
                         u8_t overwrite,
                         char *filename,
                         u8_t zip_it,
                         int argc,
                         char **argv2) {

  s32_t sn;
  qb_err_t e;
  u8_t pass;
  char *buf;
  size_t len;
  size_t zipped_len;
  u8_t *zipped;
  
  e = QB_E_OK;
  
  buf = NULL;
  len = 0;
  zipped_len = 0;
  zipped = NULL;
  
  for (pass=0; pass < 2; pass++) {
  
    const char *fmt;
    int argc_i;
    size_t sz;
    
    sz=0;
    
    if (1==pass) {
      buf[0] = '#';
    }
    sz = 1;
    
    // paste command line into TIBET file
    for (argc_i=0; argc_i < argc; argc_i++) {
    
      size_t z, alen;
      char *argvtmp;
      
      alen = strlen(argv2[argc_i]);
      argvtmp = malloc (alen + 1);
      argvtmp[alen] = '\0';
      if (NULL == argvtmp) {
        fprintf(QB_ERR, "E: Out of memory allocating TIBET CLI tmpbuf\n");
        e = QB_E_MALLOC;
        break;
      }
      for (z=0; z < alen; z++) {
        char c;
        c = argv2[argc_i][z];
        if ((c < 0x20) || (c > 0x7e)) {
          argvtmp[z] = '?';
        } else {
          argvtmp[z] = c;
        }
      }
      fmt = " %s";
      if (0 == pass) {
        sz += QB_SCPRINTF (fmt, argvtmp);
      } else {
        sz += QB_SNPRINTF (buf+sz, len-sz, len-sz, fmt, argvtmp);
      }
      free(argvtmp);
      argvtmp = NULL;
    }
    
    fmt = "\n# Rate: %d\n\n";
    
    if (0 == pass) {
      sz += QB_SCPRINTF (fmt, rate);
    } else {
      sz += QB_SNPRINTF (buf+sz, len-sz, len-sz, fmt, rate);
    }
    
    // file proper begins here
    // header sub-line:
    fmt = "\n" QB_TIBET_S_VERSION " " QB_TIBET_VERSION "\n\n";
    if (0 == pass) {
      sz += QB_SCPRINTF ("%s", fmt);
    } else {
      sz += QB_SNPRINTF (buf+sz, len-sz, len-sz, "%s", fmt);
    }
    
    for (sn=0; sn < num_spans; sn++) {
    
      qb_span_t *span;
      s64_t start;
      s32_t an;
      u16_t phase;
      s64_t smpnum;
      char *name;
      
      span = spans + sn;
    
      start = qb_get_span_start_accurate_or_rough (span);
      
      fmt = QB_TIBET_S_TIME " %lf\n";
      if (0 == pass) {
        sz += QB_SCPRINTF (fmt, ((double) start) / (double) rate);
      } else {
        sz += QB_SNPRINTF (buf+sz, len-sz, len-sz, fmt,
                           ((double) start) / (double) rate);
      }
      
      if (QB_SPAN_TYPE_LEADER == span->type) {
        fmt = QB_TIBET_S_SPEED " %lf\n" QB_TIBET_S_LEADER " %d   # (cycles) [%lld]\n\n";
        if (0 == pass) {
          sz += QB_SCPRINTF (fmt, span->speed, span->num_atoms, start);
        } else {
          sz += QB_SNPRINTF (buf+sz, len-sz, len-sz, fmt,
                             span->speed, span->num_atoms, start);
        }
      } else if (QB_SPAN_TYPE_SILENT == span->type) {
        fmt = QB_TIBET_S_SILENCE " %lf   # (seconds) [%lld]\n\n";
        if (0 == pass) {
          sz += QB_SCPRINTF (fmt, ((double) span->len / (double) rate), start);
        } else {
          sz += QB_SNPRINTF(buf+sz, len-sz, len-sz, fmt,
                            ((double) span->len / (double) rate), start);
        }
      } else if (QB_SPAN_TYPE_DATA == span->type) {
      
        if (span->detected_input_phase_ix != -1) {
          phase = qb_get_phase_for_phase_ix (span->detected_input_phase_ix);
          fmt = QB_TIBET_S_PHASE " %d   # (degrees)\n";
          if (0 == pass) {
            sz += QB_SCPRINTF (fmt, phase); //, span->speed, name);
          } else {
            sz += QB_SNPRINTF (buf+sz, len-sz, len-sz, fmt,
                               phase); //, span->speed, name);
          }
        }
        
        name = qb_is_squawk (span, (sn>0) ? (span - 1) : NULL) ? QB_TIBET_S_SQUAWK : QB_TIBET_S_DATA;
        fmt = QB_TIBET_S_SPEED " %lf\n%s\n"; // "framing 8N1"
        if (0 == pass) {
          sz += QB_SCPRINTF (fmt, span->speed, name);
        } else {
          sz += QB_SNPRINTF (buf+sz, len-sz, len-sz, fmt,
                             span->speed, name);
        }
        
        smpnum = span->atoms[0].sample_num;
        
        for (an=0; an < span->num_atoms; an++) {
          if (1 == pass) {
            if (span->atoms[an].value == '0') {
              buf[sz] = '-';
            } else {
              buf[sz] = '.';
            }
          }
          sz++;
#define QB_TIBET_LINELEN 64
          if ((an == (span->num_atoms - 1)) || ((an % QB_TIBET_LINELEN) == (QB_TIBET_LINELEN - 1))) {
            fmt = "   # [%lld] \n";
            if (0 == pass) {
              sz += QB_SCPRINTF (fmt, smpnum);
            } else {
              sz += QB_SNPRINTF(buf+sz, len-sz, len-sz, fmt,
                                smpnum);
            }
          } else if ((an % QB_TIBET_LINELEN) == 0) {
            smpnum = span->atoms[an].sample_num;
          }
        } // next cycle
        // end token
        //fmt = "end"; <- compiler complained about this  :/
        if (0 == pass) {
          sz += QB_SCPRINTF (QB_TIBET_S_END "\n\n");
        } else {
          sz += QB_SNPRINTF (buf+sz, len-sz, len-sz, QB_TIBET_S_END "\n\n");
        }
      } // endif (data span)
    } // next span
    
    if (1 == pass) {
      buf[sz] = '\n';
    }
    sz++;
    
    if (0 == pass) {
      len = sz;
      buf = qb_malloc (len);
      if (NULL == buf) {
        fprintf(QB_ERR, "E: Out of memory allocating TIBET file buffer.\n");
        e = QB_E_MALLOC;
        break;
      }
    } else {
#ifdef QB_HAVE_ZLIB
      if (zip_it) {
        printf("Compressing TIBET: %zu -> ", len);
        e = qb_zlib_compress ((u8_t *) buf, len, 1, &zipped, &zipped_len); // 1 = use gzip encoding
        printf("%zu bytes.\n", zipped_len);
      }
#endif
      if (QB_E_OK == e) {
        e = qb_write_file (filename,
                           overwrite,
                           (zip_it ? zipped     : (u8_t *) buf),
                           (zip_it ? zipped_len : len));
      }
    }
  } // next pass
  
  if (NULL != buf)    { qb_free(buf);    }
  if (NULL != zipped) { qb_free(zipped); }
  
  if (QB_E_OK == e) {
    printf("Wrote TIBET%s file (%zu bytes): %s\n",
           (zip_it ? "Z" : ""),
           (zip_it ? zipped_len : len),
           filename);
  }
  
  return e;
  
}
