/*
 *  Quadbike 2
 *  Copyright (C) 2026 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "util.h"
#include "qbio.h"

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>


#ifdef QB_HAVE_ZLIB
#include "zlib.h"
#endif

char qb_scprintf_dummy; // dummy variable; written, never read, used by QB_SCPRINTF implementation

void qb_show_meter (u8_t display_progress) {
  if (display_progress) { printf("   "); }
}

void qb_update_meter (u8_t display_progress,
                      s64_t pos,
                      s64_t len,
                      float fraction_of_total_job,
                      s32_t starting_percent) {
  s64_t val;
  if (((len / 100) == 0) || ! display_progress) {
    // avoid division by zero (or just don't display anything)
    return;
  }
  if (0 == (pos % (len / 100))) {
    printf("\b\b\b");
    val = starting_percent + (s64_t) floor((pos * 100.0f * fraction_of_total_job) / (float) len);
    if (val > 99) {
      val = 99;
    }
    printf("%2lld%%", val);
  }
  fflush(stdout); // MacOS
}

void qb_hide_meter (u8_t display_progress, u8_t newline) {
  if (display_progress) {
    printf("\b\b\b");
  }
  if (newline) { printf("\n"); }
  fflush(stdout); // MacOS
}



s16_t qb_double_to_s16 (double d) {
#ifdef QB_SANITY
  if (d > 1.0) {
    fprintf(QB_ERR, "W: %s: d > 1.0 (%lf), clamping.\n", QB_FUNC_M, d);
    d = 1.0;
  }
  if (d < -1.0) {
    fprintf(QB_ERR, "W: %s: d < -1.0 (%lf), clamping.\n", QB_FUNC_M, d);
    d = -1.0;
  }
#endif
  return (s16_t) round(d * 32767.0);
}


s16_t qb_float_to_s16 (float f) {
#ifdef QB_SANITY
  if (f > 1.0f) {
    fprintf(QB_ERR, "W: %s: f > 1.0 (%lf), clamping.\n", QB_FUNC_M, f);
    f = 1.0f;
  }
  if (f < -1.0f) {
    fprintf(QB_ERR, "W: %s: f < -1.0 (%lf), clamping.\n", QB_FUNC_M, f);
    f = -1.0f;
  }
#endif
  return (s16_t) round(f * 32767.0f);
}



//~ static double _maxX;

// configure this depending on what we
// end up doing re: normalising with max_power0,
// max_power1 or max_powerX, and call it everywhere
// we need to do that pair of calculations
//void qb_normalise_powers (double *pwr0_inout,
//                          double *pwr1_inout,
//                          double maxX) {
void qb_normalise_powers (float *pwr0_inout,
                          float *pwr1_inout,
                          float maxX) {
  (*pwr0_inout) /= maxX;
  (*pwr1_inout) /= maxX;
}

qb_err_t qb_test_rate_legal (s32_t sr) {
  u8_t a;
  a = (sr == 22050) ||
      (sr == 44100) ||
      (sr == 48000);
  if ( ! a ) {
    fprintf(QB_ERR, "W: qb_test_rate_legal: bad sample rate (%d)\n", sr);
  }
  return a ? QB_E_OK : QB_E_BAD_SAMPLE_RATE;
}

void qb_free(void *p) {
#if defined QB_ALIGNED_MALLOC && defined QB_WINDOWS
  _aligned_free(p);
#else
  free(p);
#endif
}

void *qb_malloc (size_t size) {
  size_t z;
  z = size & 0x1ff; // 2.0.4: valgrind is moaning, so ensure that we only ask for 512 pieces
  size &= ~((size_t)0x1ff);
  if (z) { size += 0x200; }
#if defined QB_ALIGNED_MALLOC // && (defined QB_VECTORS_GCC_CLANG || defined QB_VECTORS_MSVC_AVX2)
#ifdef QB_WINDOWS
  return _aligned_malloc(size, 512);
#else
  return aligned_alloc (512, size); // fix crashes with vectors using either GCC or clang on linux
#endif // not windows
#else
  return malloc(size); // MacOS vectorised code seems sane
#endif
}

void *qb_realloc (void *p, size_t size) {
#if defined QB_ALIGNED_MALLOC // && (defined QB_VECTORS_GCC_CLANG || defined QB_VECTORS_MSVC_AVX2)
#ifdef QB_WINDOWS
  return _aligned_realloc(p, size, 512);
#else
  return realloc (p, size); // fix crashes with vectors using either GCC or clang on linux
#endif // not windows
#else
  return realloc (p, size); // MacOS vectorised code seems sane
#endif
}

void qb_u32_to_little_endian (u32_t i, char buf[4]) {
  buf[0] = i         & 0xff;
  buf[1] = (i >> 8)  & 0xff;
  buf[2] = (i >> 16) & 0xff;
  buf[3] = (i >> 24) & 0xff;
//~ printf("%02x%02x%02x%02x\n", 0xff&buf[0], 0xff&buf[1], 0xff&buf[2], 0xff&buf[3]);
//~ exit(0);
}

#ifdef QB_HAVE_ZLIB
// adapted from the public domain zpipe.c:
qb_err_t qb_zlib_compress (u8_t *source,
                           size_t srclen,
                           u8_t use_gzip_encoding,
                           u8_t **dest,
                           size_t *destlen) {

  int ret, flush;
  z_stream strm;
  size_t alloced=0;
  
  // allocate deflate state
  strm.zalloc = Z_NULL;
  strm.zfree = Z_NULL;
  strm.opaque = Z_NULL;

  if (srclen > 0x7fffffff) {
    fprintf(QB_ERR, "\nB: %s: srclen is insane (%zu). Cannot continue.", QB_FUNC_M, srclen);
    return QB_E_BUG;
  }
  
  if (use_gzip_encoding) {
    ret = deflateInit2 (&strm,
                        Z_BEST_COMPRESSION, // Z_DEFAULT_COMPRESSION
                        Z_DEFLATED,
                        15 + 16, // windowbits 15, +16 for gzip encoding
                        8,
                        Z_DEFAULT_STRATEGY);
  } else {
    ret = deflateInit  (&strm, Z_BEST_COMPRESSION);
  }
  if (ret != Z_OK) {
    return QB_E_ZLIB_INIT;
  }

  // compress until end of file
  strm.avail_in = 0x7fffffff & srclen;
  flush = Z_FINISH;
  
  strm.next_in = source;

  // run deflate() on input until output buffer not full, finish
  // compression if all of source has been read in
  do {
    u8_t *dest2;
    
    if (alloced == *destlen) { // realloc if no space left
      dest2 = qb_realloc(*dest, alloced = (*destlen ? (*destlen * 2) : srclen));
      if (NULL == dest2) {
        fprintf(QB_ERR, "\nE: Out of memory enlarging zlib compress buffer.\n");
        deflateEnd(&strm);
        qb_free(*dest);
        *dest = NULL;
        return QB_E_MALLOC;
      }
      *dest = dest2;
    }
    strm.avail_out = 0x7fffffff & (alloced - *destlen); // bytes available in output buffer
    strm.next_out = *dest + *destlen;    // current offset in output buffer
//printf("a\n"); fflush(stdout);
    ret = deflate(&strm, flush);         // no bad return value
//printf("b\n"); fflush(stdout);
    //~ srcpos += srclen - strm.avail_in;
    *destlen += (alloced - *destlen) - strm.avail_out;
    
  } while (strm.avail_out == 0);
  
//printf("strm.avail_out = %d\n", strm.avail_out);
  
  // clean up
  deflateEnd(&strm);
  
  if ( ( strm.avail_in != 0 ) || ( ret != Z_STREAM_END ) ) {     // all input will be used
    if (strm.avail_in != 0) {
      fprintf(QB_ERR, "\nE: zlib compression failed -- strm.avail_in != 0.\n");
    } else {
      fprintf(QB_ERR, "\nE: zlib compression failed (code %u).\n", ret);
    }
    qb_free(*dest);
    *dest = NULL;
    return QB_E_ZLIB_COMPRESS;
  }
  
  return QB_E_OK;
  
}
#endif // QB_HAVE_ZLIB


#ifdef QB_WINDOWS
qb_err_t qb_windows_convert_utf8_string_to_wchar(char *s, LPWSTR *out, int *len_out) {
  int len;
  LPWSTR ws;

  if (NULL != len_out) {
    *len_out = 0;
  }
  *out = 0;

  // pass 1: get size
  len = MultiByteToWideChar(CP_UTF8, MB_ERR_INVALID_CHARS, s, -1, NULL, 0);
  if (0 == len) {
    return QB_E_WIN_MULTIBYTE_TO_WCHAR;
  }
  ws = qb_malloc(len * sizeof(wchar_t));
  if (NULL == ws) {
    fprintf(QB_ERR, "E: Out of memory allocating file name buffer for fopen.\n");
    return QB_E_MALLOC;
  }
  // pass 2: get string
  MultiByteToWideChar(CP_UTF8, MB_ERR_INVALID_CHARS, s, -1, ws, len);
  if (NULL != len_out) {
    *len_out = len;
  }
  *out = ws;
  return QB_E_OK;
}
#endif


qb_err_t qb_make_fully_qualified_path (char *in, char **out, char *cwd) {

#ifdef QB_WINDOWS
  //ULONG winpathlen;
  LPWSTR win_in;
  LPWSTR win_fullpath_buf;
  //int win_len;
  DWORD win_fullpath_len;
  LPWSTR win_final_file_component_within_path_lolol;
  qb_err_t e;
  int win_utf8_fullpath_len;
#else
  size_t in_len, in_len_2, out_len, cwd_len;
  size_t start;
  u8_t already_fully_qualified;
  in_len = strlen(in);
  cwd_len = strlen(cwd);
  already_fully_qualified = 0;
#endif

  *out = NULL;

#ifdef QB_WINDOWS

  e = qb_windows_convert_utf8_string_to_wchar (in, &win_in, NULL);
  if (QB_E_OK != e) { return e; }

  // pass one: get length
  win_fullpath_len = GetFullPathNameW (win_in, 0, 0, 0);
  if (0 == win_fullpath_len) {
    fprintf(QB_ERR, "E: Could not get Windows full path name for file: %s", in);
    qb_free(win_in);
    return QB_E_WIN_GET_FULL_PATH_NAME_W;
  }

  // pass two: get path
  win_fullpath_buf = qb_malloc (sizeof(WCHAR) * win_fullpath_len);
  if (NULL == win_fullpath_buf) {
    fprintf(QB_ERR, "E: Out of memory allocating win_fullpath_buf (wide) for file: %s\n", in);
    qb_free(win_in);
    return QB_E_MALLOC;
  }
  GetFullPathNameW (win_in, win_fullpath_len, win_fullpath_buf, &win_final_file_component_within_path_lolol);
  qb_free(win_in);

  // convert back to UTF-8
  win_utf8_fullpath_len = WideCharToMultiByte (CP_UTF8, 0, win_fullpath_buf, -1, NULL, 0, NULL, NULL);
  *out = qb_malloc (win_utf8_fullpath_len);
  if (NULL == *out) {
    fprintf(QB_ERR, "E: Out of memory allocating Windows UTF-8 full path for file: %s\n", in);
    qb_free(win_fullpath_buf);
    return QB_E_MALLOC;
  }
  WideCharToMultiByte(CP_UTF8, 0, win_fullpath_buf, -1, *out, win_utf8_fullpath_len, NULL, NULL);
  qb_free(win_fullpath_buf);

  if ((*out)[1] != ':') {
    fprintf(QB_ERR, "B: %s: path from GetFullPathNameW() does not start with drive letter.\n", QB_FUNC_M);
    qb_free(*out);
    *out = NULL;
    return QB_E_BUG;
  }

  // force consistent drive letter case; for some reason GetFullPathNameW()
  // doesn't force a consistent case here?

  (*out)[0] &= 0xdf;

  if ( ! isalpha ((*out)[0]) ) {
    fprintf(QB_ERR, "B: %s: path from GetFullPathNameW() does not start with drive letter [2].\n", QB_FUNC_M);
    qb_free(*out);
    *out = NULL;
    return QB_E_BUG;
  }



#else
  already_fully_qualified = (QB_PATHSEP == in[0]); // Unix: path starts with '/'
//#endif

  // already fully qualified?
  if (already_fully_qualified) {
    *out = qb_malloc(in_len+1);
    if (NULL == *out) {
      fprintf(QB_ERR, "E: Out of memory allocating fully qualified path.\n");
      return QB_E_MALLOC;
    }
    // already qualified, so just copy it
    memcpy(*out, in, in_len+1);
//printf("FQP (len %zu): %s\n", strlen(*out), *out);
    return QB_E_OK;
  } // done
  
  /*
  // skip any and all leading path separators
  start = 0;
  while ((start < in_len) && (QB_PATHSEP == in[start])) {
printf("start = %zu\n", start);
    start++;
  }
  if (start == in_len) {
    // path was entirely composed of separators
    fprintf(QB_ERR, "E: Path was made entirely of tin: %s\n", in);
    return QB_E_PATH_MADE_ENTIRELY_OF_TIN;
  }
  in_len_2 = in_len - start;
  */
  
  in_len_2 = in_len;
  start = 0;
  
  // malloc it
  out_len = in_len_2 + 1 + cwd_len;
  *out = qb_malloc (out_len + 1);
  if (NULL == *out) {
    fprintf(QB_ERR, "E: Out of memory allocating fully qualified path [2].\n");
    return QB_E_MALLOC;
  }
  
  // copy
  memcpy(*out, cwd, cwd_len);
  (*out)[cwd_len] = QB_PATHSEP;
  memcpy(*out + cwd_len + 1, in + start, in_len_2 + 1);
  
//printf("FQP (len %zu): %s\n", out_len, *out);

#endif // not windows
  
  return QB_E_OK;
  
}

#ifndef QB_WINDOWS
#include <unistd.h>
#endif


// Unixes only; does nothing on Windows
qb_err_t qb_get_working_directory (char **out) {
  *out = NULL;
#ifndef QB_WINDOWS
  *out = getcwd(NULL, 0);
#endif
  return QB_E_OK;
}
//#endif // not windows

const char *qb_util_basename (const char *path) {
  size_t len, z;
  // 2.0.4: basename
  len = strlen(path);
  for (z=len; z>0; z--) {
    if (QB_PATHSEP == path[z-1]) {
      break;
    }
  }
  return path+z;
}
