/*
 *  Quadbike 2
 *  Copyright (C) 2026 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#ifndef QB_VECTOR_H
#define QB_VECTOR_H

#include "build.h"
#include "qb_types.h"

#if defined QB_VECTORS_GCC_CLANG

// for _GCC_CLANG you can change this and everything should just work:
//#define QB_VECSIZE 8   // must be a power of two

typedef float  qb_vec_f_t __attribute__ ((vector_size (QB_VECSIZE * 4)));
//typedef double qb_vec_d_t __attribute__ ((vector_size (QB_VECSIZE * 8)));
typedef int    qb_vec_i_t __attribute__ ((vector_size (QB_VECSIZE * 4)));
#elif defined QB_VECTORS_MSVC_AVX2

#include <immintrin.h>

// for _MSVC_AVX2 this is fixed by the use of __m256i, and must not be changed
#define QB_VECSIZE 8

// Visual Studio: AVX2 only
typedef __m256  qb_vec_f_t;   // 256 bits; space for 8 floats
typedef __m256i qb_vec_i_t;   // 256 bits; space for 8 ints

#elif defined QB_VECTORS_MSVC_AVX512

#include <immintrin.h>

// for _MSVC_AVX512 this is fixed by the use of __m512i, and must not be changed
#define QB_VECSIZE 16

// Visual Studio: AVX2/512 only
typedef __m512  qb_vec_f_t;   // 512 bits; space for 16 floats
typedef __m512i qb_vec_i_t;   // 512 bits; space for 16 ints

#endif // defined MSVC_AVX512


#if defined QB_VECTORS_GCC_CLANG || defined QB_VECTORS_MSVC_AVX2 || defined QB_VECTORS_MSVC_AVX512

#define QB_VECTYPE_INVALID 0
#define QB_VECTYPE_FLOAT   1
//#define QB_VECTYPE_DOUBLE  2 // not used

typedef struct qb_vec_buf_s {
  u8_t type; // QB_VECTYPE_FLOAT, QB_VECTYPE_DOUBLE
  s64_t alloc;
  // index of column corresponding to [0] in original signal
  // once filter delays etc. are taken into account
  s64_t zero_ix;
  // length of the "main piece" of data, i.e. the non-overlapping part,
  // which is (linear len / QB_VECSIZE), rounded up
  s64_t piece_len;
  s64_t linear_len;
  union {
    qb_vec_f_t *f;
    //qb_vec_d_t *d; // never actually used
  } v;
} qb_vec_buf_t;

qb_err_t qb_vec_buf_init (qb_vec_buf_t *v,
                          u8_t vectype,
                          void *data, // may be NULL
                          s64_t linear_len,
                          s64_t pre_overlap,
                          s64_t post_overlap,
                          u8_t display_progress,
                          char *indent_s);
                      
void qb_vec_buf_finish (qb_vec_buf_t *v);

qb_err_t qb_vec_buf_unswizzle (qb_vec_buf_t *v,
                               float **f_out,
                               //double **d_out,
                               u8_t dp, // display_progress
                               char *indent,
                               char *extra_text);
                               
qb_err_t qb_vec_buf_unswizzle_to_s16 (qb_vec_buf_t *v, s16_t **out, float scaler);

//void qb_vec_buf_debug_print (qb_vec_buf_t *v);

#endif // have vectors

#endif // QB_VECTOR_H
